<?php
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

$report_type = isset($_GET['report_type']) ? $_GET['report_type'] : 'flights';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');

function getOperationalReport($conn, $report_type, $date_from, $date_to) {
    switch($report_type) {
        case 'flights':
            return $conn->query("
                SELECT 
                    f.flight_number,
                    a1.airport_code AS origin,
                    a2.airport_code AS destination,
                    COUNT(f.flight_id) AS total_flights,
                    AVG(f.available_seats) AS avg_available_seats,
                    COUNT(CASE WHEN f.status = 'cancelled' THEN 1 END) AS cancelled_flights,
                    COUNT(CASE WHEN f.status = 'completed' THEN 1 END) AS completed_flights
                FROM flights f
                JOIN airports a1 ON f.origin_id = a1.airport_id
                JOIN airports a2 ON f.destination_id = a2.airport_id
                WHERE f.departure_time BETWEEN '$date_from' AND '$date_to'
                GROUP BY f.flight_number, a1.airport_code, a2.airport_code"
            );

        case 'routes':
            return $conn->query("
                SELECT 
                    a1.airport_code AS origin,
                    a1.city AS origin_city,
                    a2.airport_code AS destination,
                    a2.city AS destination_city,
                    COUNT(DISTINCT f.flight_number) AS route_frequency,
                    AVG(f.price) AS average_price,
                    COUNT(b.booking_id) AS total_bookings
                FROM flights f
                JOIN airports a1 ON f.origin_id = a1.airport_id
                JOIN airports a2 ON f.destination_id = a2.airport_id
                LEFT JOIN bookings b ON f.flight_id = b.flight_id
                WHERE f.departure_time BETWEEN '$date_from' AND '$date_to'
                GROUP BY a1.airport_code, a2.airport_code"
            );

        case 'support':
            return $conn->query("
                SELECT 
                    st.status,
                    COUNT(*) as ticket_count,
                    st.priority,
                    AVG(TIMESTAMPDIFF(HOUR, st.created_at, st.updated_at)) as avg_resolution_time,
                    COUNT(CASE WHEN tc.comment_id IS NOT NULL THEN 1 END) as total_comments
                FROM support_tickets st
                LEFT JOIN ticket_comments tc ON st.ticket_id = tc.ticket_id
                WHERE st.created_at BETWEEN '$date_from' AND '$date_to'
                GROUP BY st.status, st.priority"
            );

        case 'expenses':
            return $conn->query("
                SELECT 
                    expense_type,
                    COUNT(*) as transaction_count,
                    SUM(amount) as total_amount,
                    AVG(amount) as average_amount,
                    MIN(amount) as min_amount,
                    MAX(amount) as max_amount
                FROM expenses
                WHERE created_at BETWEEN '$date_from' AND '$date_to'
                GROUP BY expense_type
                ORDER BY total_amount DESC"
            );
    }
}

$result = getOperationalReport($conn, $report_type, $date_from, $date_to);
?>

<!doctype html>
<html lang="en" data-layout="vertical" data-topbar="light" data-sidebar="dark" data-sidebar-size="lg">
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Operational Reports')); ?>
    <?php include 'layouts/head-css.php'; ?>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Reports', 'title' => 'Operational Reports')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header d-flex justify-content-between align-items-center">
                                    <h4 class="card-title mb-0">Operational Reports</h4>
                                    <button onclick="exportToCSV()" class="btn btn-success">Export to CSV</button>
                                </div>
                                <div class="card-body">
                                    <form class="row g-3 mb-4" method="GET">
                                        <div class="col-md-3">
                                            <select class="form-select" name="report_type">
                                                <option value="flights" <?php echo $report_type == 'flights' ? 'selected' : ''; ?>>Flight Operations</option>
                                                <option value="routes" <?php echo $report_type == 'routes' ? 'selected' : ''; ?>>Route Analysis</option>
                                                <option value="support" <?php echo $report_type == 'support' ? 'selected' : ''; ?>>Support Tickets</option>
                                                <option value="expenses" <?php echo $report_type == 'expenses' ? 'selected' : ''; ?>>Expense Analysis</option>
                                            </select>
                                        </div>
                                        <div class="col-md-3">
                                            <input type="date" class="form-control" name="date_from" 
                                                   value="<?php echo htmlspecialchars($date_from); ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <input type="date" class="form-control" name="date_to" 
                                                   value="<?php echo htmlspecialchars($date_to); ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <button type="submit" class="btn btn-primary w-100">Generate Report</button>
                                        </div>
                                    </form>

                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <?php if ($report_type == 'flights'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Flight Number</th>
                                                        <th>Route</th>
                                                        <th>Total Flights</th>
                                                        <th>Avg Available Seats</th>
                                                        <th>Completed</th>
                                                        <th>Cancelled</th>
                                                        <th>Completion Rate</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['flight_number']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['origin'] . ' - ' . $row['destination']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_flights']); ?></td>
                                                            <td><?php echo round($row['avg_available_seats']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['completed_flights']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['cancelled_flights']); ?></td>
                                                            <td><?php echo number_format(($row['completed_flights'] / $row['total_flights']) * 100, 1) . '%'; ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'routes'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Origin</th>
                                                        <th>Destination</th>
                                                        <th>Route Frequency</th>
                                                        <th>Average Price</th>
                                                        <th>Total Bookings</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['origin'] . ' (' . $row['origin_city'] . ')'); ?></td>
                                                            <td><?php echo htmlspecialchars($row['destination'] . ' (' . $row['destination_city'] . ')'); ?></td>
                                                            <td><?php echo htmlspecialchars($row['route_frequency']); ?></td>
                                                            <td><?php echo number_format($row['average_price'], 2); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_bookings']); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'support'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Status</th>
                                                        <th>Priority</th>
                                                        <th>Ticket Count</th>
                                                        <th>Avg Resolution Time (hrs)</th>
                                                        <th>Total Comments</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['status']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['priority']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['ticket_count']); ?></td>
                                                            <td><?php echo round($row['avg_resolution_time'], 1); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_comments']); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'expenses'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Expense Type</th>
                                                        <th>Transaction Count</th>
                                                        <th>Total Amount</th>
                                                        <th>Average Amount</th>
                                                        <th>Min Amount</th>
                                                        <th>Max Amount</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['expense_type']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['transaction_count']); ?></td>
                                                            <td><?php echo number_format($row['total_amount'], 2); ?></td>
                                                            <td><?php echo number_format($row['average_amount'], 2); ?></td>
                                                            <td><?php echo number_format($row['min_amount'], 2); ?></td>
                                                            <td><?php echo number_format($row['max_amount'], 2); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>
                                            <?php endif; ?>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    
    <script>
    function exportToCSV() {
        const urlParams = new URLSearchParams(window.location.search);
        const params = {
            report_type: urlParams.get('report_type') || 'flights',
            date_from: urlParams.get('date_from') || '',
            date_to: urlParams.get('date_to') || ''
        };
        window.location.href = `export_operational_report.php?${new URLSearchParams(params)}`;
    }
    </script>
</body>
</html>