<?php
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

$report_type = isset($_GET['report_type']) ? $_GET['report_type'] : 'revenue';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');

function getReportData($conn, $report_type, $date_from, $date_to) {
    switch($report_type) {
        case 'revenue':
            return $conn->query("
                SELECT 
                    a1.airport_code AS origin,
                    a2.airport_code AS destination,
                    COUNT(b.booking_id) AS total_bookings,
                    SUM(b.total_amount) AS total_revenue,
                    AVG(b.total_amount) AS average_booking_value
                FROM bookings b
                JOIN flights f ON b.flight_id = f.flight_id
                JOIN airports a1 ON f.origin_id = a1.airport_id
                JOIN airports a2 ON f.destination_id = a2.airport_id
                WHERE b.status = 'Booked' 
                AND b.booking_date BETWEEN '$date_from' AND '$date_to'
                GROUP BY a1.airport_code, a2.airport_code
                ORDER BY total_revenue DESC"
            );

        case 'bookings':
            return $conn->query("
                SELECT 
                    f.flight_number,
                    a1.airport_code AS origin,
                    a2.airport_code AS destination,
                    f.capacity,
                    f.available_seats,
                    ((f.capacity - f.available_seats) / f.capacity * 100) AS occupancy_rate,
                    COUNT(b.booking_id) AS total_bookings
                FROM flights f
                JOIN airports a1 ON f.origin_id = a1.airport_id
                JOIN airports a2 ON f.destination_id = a2.airport_id
                LEFT JOIN bookings b ON f.flight_id = b.flight_id
                WHERE f.departure_time BETWEEN '$date_from' AND '$date_to'
                GROUP BY f.flight_id"
            );

        case 'customers':
            return $conn->query("
                SELECT 
                    c.first_name,
                    c.last_name,
                    COUNT(b.booking_id) AS total_bookings,
                    SUM(b.total_amount) AS total_spent,
                    AVG(b.total_amount) AS avg_booking_value,
                    MAX(b.booking_date) AS last_booking_date
                FROM customers c
                LEFT JOIN bookings b ON c.customer_id = b.customer_id
                WHERE b.booking_date BETWEEN '$date_from' AND '$date_to'
                GROUP BY c.customer_id
                ORDER BY total_spent DESC"
            );

        case 'payments':
            return $conn->query("
                SELECT 
                    DATE_FORMAT(b.booking_date, '%Y-%m') AS month,
                    COUNT(b.booking_id) AS total_bookings,
                    SUM(b.total_amount) AS total_booking_value,
                    SUM(CASE WHEN b.payment_status = 'Fully Paid' 
                        THEN b.total_amount ELSE 0 END) AS collected_amount,
                    SUM(CASE WHEN b.payment_status = 'Pending' 
                        THEN b.total_amount ELSE 0 END) AS pending_amount
                FROM bookings b
                WHERE b.booking_date BETWEEN '$date_from' AND '$date_to'
                GROUP BY month"
            );
    }
}

$result = getReportData($conn, $report_type, $date_from, $date_to);
?>

<!doctype html>
<html lang="en" data-layout="vertical" data-topbar="light" data-sidebar="dark" data-sidebar-size="lg">
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Financial Reports')); ?>
    <?php include 'layouts/head-css.php'; ?>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Reports', 'title' => 'Financial Reports')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header d-flex justify-content-between align-items-center">
                                    <h4 class="card-title mb-0">Financial Reports</h4>
                                    <button onclick="exportToCSV()" class="btn btn-success">Export to CSV</button>
                                </div>
                                <div class="card-body">
                                    <form class="row g-3 mb-4" method="GET">
                                        <div class="col-md-3">
                                            <select class="form-select" name="report_type">
                                                <option value="revenue" <?php echo $report_type == 'revenue' ? 'selected' : ''; ?>>Revenue by Route</option>
                                                <option value="bookings" <?php echo $report_type == 'bookings' ? 'selected' : ''; ?>>Booking Statistics</option>
                                                <option value="customers" <?php echo $report_type == 'customers' ? 'selected' : ''; ?>>Customer Analytics</option>
                                                <option value="payments" <?php echo $report_type == 'payments' ? 'selected' : ''; ?>>Payment Collection</option>
                                            </select>
                                        </div>
                                        <div class="col-md-3">
                                            <input type="date" class="form-control" name="date_from" 
                                                   value="<?php echo htmlspecialchars($date_from); ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <input type="date" class="form-control" name="date_to" 
                                                   value="<?php echo htmlspecialchars($date_to); ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <button type="submit" class="btn btn-primary w-100">Generate Report</button>
                                        </div>
                                    </form>

                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <?php if ($report_type == 'revenue'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Origin</th>
                                                        <th>Destination</th>
                                                        <th>Total Bookings</th>
                                                        <th>Total Revenue</th>
                                                        <th>Average Booking Value</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['origin']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['destination']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_bookings']); ?></td>
                                                            <td><?php echo number_format($row['total_revenue'], 2); ?></td>
                                                            <td><?php echo number_format($row['average_booking_value'], 2); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'bookings'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Flight Number</th>
                                                        <th>Route</th>
                                                        <th>Capacity</th>
                                                        <th>Available Seats</th>
                                                        <th>Occupancy Rate</th>
                                                        <th>Total Bookings</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['flight_number']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['origin'] . ' - ' . $row['destination']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['capacity']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['available_seats']); ?></td>
                                                            <td><?php echo number_format($row['occupancy_rate'], 1) . '%'; ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_bookings']); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'customers'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Customer Name</th>
                                                        <th>Total Bookings</th>
                                                        <th>Total Spent</th>
                                                        <th>Average Booking Value</th>
                                                        <th>Last Booking Date</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['first_name'] . ' ' . $row['last_name']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_bookings']); ?></td>
                                                            <td><?php echo number_format($row['total_spent'], 2); ?></td>
                                                            <td><?php echo number_format($row['avg_booking_value'], 2); ?></td>
                                                            <td><?php echo date('d M Y', strtotime($row['last_booking_date'])); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'payments'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Month</th>
                                                        <th>Total Bookings</th>
                                                        <th>Total Value</th>
                                                        <th>Collected Amount</th>
                                                        <th>Pending Amount</th>
                                                        <th>Collection Rate</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo date('M Y', strtotime($row['month'] . '-01')); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_bookings']); ?></td>
                                                            <td><?php echo number_format($row['total_booking_value'], 2); ?></td>
                                                            <td><?php echo number_format($row['collected_amount'], 2); ?></td>
                                                            <td><?php echo number_format($row['pending_amount'], 2); ?></td>
                                                            <td><?php echo number_format(($row['collected_amount'] / $row['total_booking_value']) * 100, 1) . '%'; ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>
                                            <?php endif; ?>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    
    <script>
    function exportToCSV() {
        const urlParams = new URLSearchParams(window.location.search);
        const params = {
            report_type: urlParams.get('report_type') || 'revenue',
            date_from: urlParams.get('date_from') || '',
            date_to: urlParams.get('date_to') || ''
        };
        window.location.href = `export_report.php?${new URLSearchParams(params)}`;
    }
    </script>
</body>
</html>