<?php
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

$applicationId = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Fetch visa application details
$stmt = $conn->prepare("
    SELECT v.*, u.full_name as agent_name
    FROM visa_applications v
    LEFT JOIN users u ON v.assigned_agent_id = u.user_id
    WHERE v.application_id = ? AND v.status IN ('Submitted', 'Under Review')
");

$stmt->bind_param("i", $applicationId);
$stmt->execute();
$result = $stmt->get_result();
$visa = $result->fetch_assoc();

if (!$visa) {
    header('Location: all_visas.php');
    exit();
}

// Fetch agents for assignment
$agentsQuery = "SELECT user_id, full_name FROM users WHERE user_role IN ('admin', 'operations') ORDER BY full_name";
$agents = $conn->query($agentsQuery)->fetch_all(MYSQLI_ASSOC);
?>

<!doctype html>
<html lang="en" data-layout="vertical" data-topbar="light" data-sidebar="dark" data-sidebar-size="lg">
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Edit Visa Application')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <link href="assets/libs/sweetalert2/sweetalert2.min.css" rel="stylesheet">
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <div class="row">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Edit Visa Application</h4>
                                </div>
                                <div class="card-body">
                                    <form id="editVisaForm" method="POST" action="update_visa.php" enctype="multipart/form-data">
                                        <input type="hidden" name="application_id" value="<?php echo $applicationId; ?>">
                                        <div class="row g-3">
                                            <!-- Personal Information -->
                                            <div class="col-md-4">
                                                <label class="form-label">First Name</label>
                                                <input type="text" class="form-control" name="first_name" 
                                                       value="<?php echo htmlspecialchars($visa['first_name']); ?>" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Last Name</label>
                                                <input type="text" class="form-control" name="last_name" 
                                                       value="<?php echo htmlspecialchars($visa['last_name']); ?>" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Gender</label>
                                                <select class="form-select" name="gender" required>
                                                    <option value="Male" <?php echo $visa['gender'] == 'Male' ? 'selected' : ''; ?>>Male</option>
                                                    <option value="Female" <?php echo $visa['gender'] == 'Female' ? 'selected' : ''; ?>>Female</option>
                                                    <option value="Other" <?php echo $visa['gender'] == 'Other' ? 'selected' : ''; ?>>Other</option>
                                                </select>
                                            </div>

                                            <div class="col-md-4">
                                                <label class="form-label">Date of Birth</label>
                                                <input type="date" class="form-control" name="date_of_birth" 
                                                       value="<?php echo $visa['date_of_birth']; ?>" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Passport Number</label>
                                                <input type="text" class="form-control" name="passport_number" 
                                                       value="<?php echo htmlspecialchars($visa['passport_number']); ?>" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Nationality</label>
                                                <input type="text" class="form-control" name="nationality" 
                                                       value="<?php echo htmlspecialchars($visa['nationality']); ?>" required>
                                            </div>

                                            <!-- Contact Information -->
                                            <div class="col-md-6">
                                                <label class="form-label">Email</label>
                                                <input type="email" class="form-control" name="email" 
                                                       value="<?php echo htmlspecialchars($visa['email']); ?>" required>
                                            </div>
                                            <div class="col-md-6">
                                                <label class="form-label">Phone Number</label>
                                                <input type="tel" class="form-control" name="phone_number" 
                                                       value="<?php echo htmlspecialchars($visa['phone_number']); ?>" required>
                                            </div>

                                            <!-- Visa Details -->
                                            <div class="col-md-4">
                                                <label class="form-label">Destination Country</label>
                                                <input type="text" class="form-control" name="destination_country" 
                                                       value="<?php echo htmlspecialchars($visa['destination_country']); ?>" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Application Type</label>
                                                <select class="form-select" name="application_type" required>
                                                    <?php
                                                    $types = ['Tourist', 'Business', 'Student', 'Work', 'Transit'];
                                                    foreach ($types as $type) {
                                                        $selected = ($visa['application_type'] === $type) ? 'selected' : '';
                                                        echo "<option value=\"$type\" $selected>$type</option>";
                                                    }
                                                    ?>
                                                </select>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Entry Type</label>
                                                <select class="form-select" name="entry_type">
                                                    <option value="Single" <?php echo $visa['entry_type'] == 'Single' ? 'selected' : ''; ?>>Single Entry</option>
                                                    <option value="Multiple" <?php echo $visa['entry_type'] == 'Multiple' ? 'selected' : ''; ?>>Multiple Entry</option>
                                                </select>
                                            </div>

                                            <!-- Financial Details -->
                                            <div class="col-md-4">
                                                <label class="form-label">Processing Fee (USD)</label>
                                                <input type="number" step="0.01" class="form-control" name="processing_fee" 
                                                       value="<?php echo $visa['processing_fee']; ?>" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Monthly Income (USD)</label>
                                                <input type="number" step="0.01" class="form-control" name="monthly_income" 
                                                       value="<?php echo $visa['monthly_income']; ?>">
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Total Trip Cost (USD)</label>
                                                <input type="number" step="0.01" class="form-control" name="total_trip_cost" 
                                                       value="<?php echo $visa['total_trip_cost']; ?>">
                                            </div>

                                            <!-- Travel Dates -->
                                            <div class="col-md-6">
                                                <label class="form-label">Intended Arrival Date</label>
                                                <input type="date" class="form-control" name="intended_arrival_date" 
                                                       value="<?php echo $visa['intended_arrival_date']; ?>" required>
                                            </div>
                                            <div class="col-md-6">
                                                <label class="form-label">Intended Departure Date</label>
                                                <input type="date" class="form-control" name="intended_departure_date" 
                                                       value="<?php echo $visa['intended_departure_date']; ?>" required>
                                            </div>

                                            <!-- Documents -->
                                            <div class="col-md-4">
                                                <label class="form-label">Passport Copy</label>
                                                <input type="file" class="form-control" name="passport_copy">
                                                <?php if ($visa['passport_copy_path']): ?>
                                                    <small class="text-muted">Current file: <a href="<?php echo $visa['passport_copy_path']; ?>" target="_blank">View</a></small>
                                                <?php endif; ?>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Photo</label>
                                                <input type="file" class="form-control" name="photo">
                                                <?php if ($visa['photo_path']): ?>
                                                    <small class="text-muted">Current file: <a href="<?php echo $visa['photo_path']; ?>" target="_blank">View</a></small>
                                                <?php endif; ?>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">Travel Insurance Proof</label>
                                                <input type="file" class="form-control" name="travel_insurance_proof">
                                                <?php if ($visa['travel_insurance_proof_path']): ?>
                                                    <small class="text-muted">Current file: <a href="<?php echo $visa['travel_insurance_proof_path']; ?>" target="_blank">View</a></small>
                                                <?php endif; ?>
                                            </div>

                                            <!-- Status and Agent -->
                                            <div class="col-md-6">
                                                <label class="form-label">Status</label>
                                                <select class="form-select" name="status" required>
                                                    <?php
                                                    $statuses = ['Submitted', 'Under Review', 'Additional Documents Required', 'Approved', 'Rejected'];
                                                    foreach ($statuses as $status) {
                                                        $selected = ($visa['status'] === $status) ? 'selected' : '';
                                                        echo "<option value=\"$status\" $selected>$status</option>";
                                                    }
                                                    ?>
                                                </select>
                                            </div>
                                            <div class="col-md-6">
                                                <label class="form-label">Assigned Agent</label>
                                                <select class="form-select" name="assigned_agent_id">
                                                    <option value="">Select Agent</option>
                                                    <?php foreach ($agents as $agent): ?>
                                                        <?php $selected = ($visa['assigned_agent_id'] == $agent['user_id']) ? 'selected' : ''; ?>
                                                        <option value="<?php echo $agent['user_id']; ?>" <?php echo $selected; ?>>
                                                            <?php echo htmlspecialchars($agent['full_name']); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>

                                            <!-- Submit Buttons -->
                                            <div class="col-12">
                                                <button type="submit" class="btn btn-primary">Update Application</button>
                                                <a href="all_visas.php" class="btn btn-secondary">Cancel</a>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="assets/libs/sweetalert2/sweetalert2.min.js"></script>

    <script>
    document.getElementById('editVisaForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);

        fetch('update_visa.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                Swal.fire({
                    title: 'Success!',
                    text: data.message,
                    icon: 'success',
                    confirmButtonText: 'OK'
                }).then((result) => {
                    if (result.isConfirmed) {
                        window.location.href = 'all_visas.php';
                    }
                });
            } else {
                Swal.fire({
                    title: 'Error!',
                    text: data.message,
                    icon: 'error',
                    confirmButtonText: 'OK'
                });
            }
        })
        .catch(error => {
            console.error('Error:', error);
            Swal.fire({
                title: 'Error!',
                text: 'An error occurred while updating the application.',
                icon: 'error',
                confirmButtonText: 'OK'
            });
        });
    });
    </script>
</body>
</html>