<?php
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

function getAgencyEarnings($conn, $date_from, $date_to) {
    // Ticket earnings
    $ticket_earnings = $conn->query("
        SELECT 
            DATE_FORMAT(created_date, '%Y-%m') as month,
            COUNT(*) as ticket_count,
            SUM(agency_fee) as ticket_fees,
            SUM(ticket_cost) as ticket_sales,
            booking_class,
            airline
        FROM ticket_details
        WHERE created_date BETWEEN '$date_from' AND '$date_to'
        GROUP BY month, booking_class, airline
        ORDER BY month DESC
    ");

    // Cargo earnings
    $cargo_earnings = $conn->query("
        SELECT 
            DATE_FORMAT(booking_date, '%Y-%m') as month,
            COUNT(*) as cargo_count,
            SUM(total_amount) as total_revenue,
            cargo_type,
            currency
        FROM cargo_bookings
        WHERE booking_date BETWEEN '$date_from' AND '$date_to'
        GROUP BY month, cargo_type, currency
        ORDER BY month DESC
    ");

    // Visa earnings
    $visa_earnings = $conn->query("
        SELECT 
            DATE_FORMAT(created_at, '%Y-%m') as month,
            COUNT(*) as visa_count,
            payment,
            SUM(commission) as processing_fees,
            SUM(commission) as service_charges
        FROM visa_applications
        WHERE created_at BETWEEN '$date_from' AND '$date_to'
        GROUP BY month, commission
        ORDER BY month DESC
    ");

    return [
        'tickets' => $ticket_earnings,
        'cargo' => $cargo_earnings,
        'visa' => $visa_earnings
    ];
}

$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$results = getAgencyEarnings($conn, $date_from, $date_to);
?>

<!doctype html>
<html lang="en" data-layout="vertical" data-topbar="light" data-sidebar="dark" data-sidebar-size="lg">
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Agency Earnings Report')); ?>
    <?php include 'layouts/head-css.php'; ?>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Reports', 'title' => 'Agency Earnings')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header d-flex justify-content-between align-items-center">
                                    <h4 class="card-title mb-0">Agency Earnings Report</h4>
                                    <div>
                                        <button onclick="printReport()" class="btn btn-info me-2">Print</button>
                                        <button onclick="exportToCSV()" class="btn btn-success">Export to CSV</button>
                                    </div>
                                </div>
                                <div class="card-body">
                                    <form class="row g-3 mb-4" method="GET">
                                        <div class="col-md-4">
                                            <input type="date" class="form-control" name="date_from" 
                                                   value="<?php echo htmlspecialchars($date_from); ?>">
                                        </div>
                                        <div class="col-md-4">
                                            <input type="date" class="form-control" name="date_to" 
                                                   value="<?php echo htmlspecialchars($date_to); ?>">
                                        </div>
                                        <div class="col-md-4">
                                            <button type="submit" class="btn btn-primary w-100">Generate Report</button>
                                        </div>
                                    </form>

                                    <!-- Ticket Earnings -->
                                    <h5 class="mt-4">Ticket Agency Fees</h5>
                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <thead>
                                                <tr>
                                                    <th>Month</th>
                                                    <th>Airline</th>
                                                    <th>Booking Class</th>
                                                    <th>Tickets</th>
                                                    <th>Ticket Sales</th>
                                                    <th>Agency Fees</th>
                                                    <th>Fee %</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php while ($row = $results['tickets']->fetch_assoc()): ?>
                                                    <tr>
                                                        <td><?php echo date('M Y', strtotime($row['month'] . '-01')); ?></td>
                                                        <td><?php echo htmlspecialchars($row['airline']); ?></td>
                                                        <td><?php echo htmlspecialchars($row['booking_class']); ?></td>
                                                        <td><?php echo htmlspecialchars($row['ticket_count']); ?></td>
                                                        <td><?php echo number_format($row['ticket_sales'], 2); ?></td>
                                                        <td><?php echo number_format($row['ticket_fees'], 2); ?></td>
                                                        <td><?php echo number_format(($row['ticket_fees'] / $row['ticket_sales']) * 100, 1) . '%'; ?></td>
                                                    </tr>
                                                <?php endwhile; ?>
                                            </tbody>
                                        </table>
                                    </div>

                                    <!-- Cargo Earnings -->
                                    <h5 class="mt-4">Cargo Revenue</h5>
                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <thead>
                                                <tr>
                                                    <th>Month</th>
                                                    <th>Cargo Type</th>
                                                    <th>Bookings</th>
                                                    <th>Total Revenue</th>
                                                    <th>Currency</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php while ($row = $results['cargo']->fetch_assoc()): ?>
                                                    <tr>
                                                        <td><?php echo date('M Y', strtotime($row['month'] . '-01')); ?></td>
                                                        <td><?php echo htmlspecialchars($row['cargo_type']); ?></td>
                                                        <td><?php echo htmlspecialchars($row['cargo_count']); ?></td>
                                                        <td><?php echo number_format($row['total_revenue'], 2); ?></td>
                                                        <td><?php echo htmlspecialchars($row['currency']); ?></td>
                                                    </tr>
                                                <?php endwhile; ?>
                                            </tbody>
                                        </table>
                                    </div>

                                    <!-- Visa Earnings -->
                                    <h5 class="mt-4">Visa Processing Fees</h5>
                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <thead>
                                                <tr>
                                                    <th>Month</th>
                                                    <th>Destination</th>
                                                    <th>Visa Type</th>
                                                    <th>Applications</th>
                                                    <th>Processing Fees</th>
                                                    <th>Service Charges</th>
                                                    <th>Total Revenue</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php while ($row = $results['visa']->fetch_assoc()): ?>
                                                    <tr>
                                                        <td><?php echo date('M Y', strtotime($row['month'] . '-01')); ?></td>
                                                        <td><?php echo htmlspecialchars($row['destination_country']); ?></td>
                                                        <td><?php echo htmlspecialchars($row['application_type']); ?></td>
                                                        <td><?php echo htmlspecialchars($row['visa_count']); ?></td>
                                                        <td><?php echo number_format($row['processing_fees'], 2); ?></td>
                                                        <td><?php echo number_format($row['service_charges'], 2); ?></td>
                                                        <td><?php echo number_format($row['processing_fees'] + $row['service_charges'], 2); ?></td>
                                                    </tr>
                                                <?php endwhile; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="assets/js/app.js"></script>
    <script>
    function exportToCSV() {
        const urlParams = new URLSearchParams(window.location.search);
        const params = {
            date_from: urlParams.get('date_from') || '',
            date_to: urlParams.get('date_to') || ''
        };
        window.location.href = `export_agency_earnings.php?${new URLSearchParams(params)}`;
    }

    function printReport() {
        window.print();
    }
    </script>
</body>
</html>