<?php
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

// Check if ticket ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header('Location: all_cancellations.php');
    exit();
}

$ticket_id = intval($_GET['id']);

// Fetch ticket details
$query = "SELECT * FROM ticket_details WHERE ticket_id = ? AND status = 'Cancelled'";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $ticket_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header('Location: all_cancellations.php');
    exit();
}

$ticket = $result->fetch_assoc();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $conn->begin_transaction();
    try {
        // Get form data
        $new_ticket_ref = $conn->real_escape_string($_POST['ticket_ref']);
        $booking_class = $conn->real_escape_string($_POST['booking_class']);
        $airline = $conn->real_escape_string($_POST['airline']);
        $flight_route = $conn->real_escape_string($_POST['flight_route']);
        $flight_date = $conn->real_escape_string($_POST['flight_date']);
        $ticket_cost = floatval($_POST['ticket_cost']);
        $agency_fee = floatval($_POST['agency_fee']);
        $currency = $conn->real_escape_string($_POST['currency']);
        $notes = $conn->real_escape_string($_POST['notes']);
        
        // Create new ticket entry
        $insert_query = "INSERT INTO ticket_details (
            salutation, customer_name, ticket_ref, booking_class, 
            airline, flight_route, flight_date, ticket_cost, agency_fee, 
            currency, status, created_by, created_date, updated_date
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Active', ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)";
        
        $insert_stmt = $conn->prepare($insert_query);
        $insert_stmt->bind_param(
            'sssssssddsi',
            $ticket['salutation'],
            $ticket['customer_name'],
            $new_ticket_ref,
            $booking_class,
            $airline,
            $flight_route,
            $flight_date,
            $ticket_cost,
            $agency_fee,
            $currency,
            $_SESSION['user_id']
        );
        
        if (!$insert_stmt->execute()) {
            throw new Exception("Failed to create new ticket");
        }
        
        $new_ticket_id = $conn->insert_id;
        
        // Add audit log
        $log_query = "INSERT INTO audit_logs (user_id, action, log_details, ip_address, user_agent) 
                      VALUES (?, 'REISSUE_TICKET', ?, ?, ?)";
        $log_stmt = $conn->prepare($log_query);
        $log_details = "Reissued ticket ID: " . $ticket_id . " as new ticket ID: " . $new_ticket_id . 
                      " with reference: " . $new_ticket_ref . ". Notes: " . $notes;
        $log_stmt->bind_param(
            'isss',
            $_SESSION['user_id'],
            $log_details,
            $_SERVER['REMOTE_ADDR'],
            $_SERVER['HTTP_USER_AGENT']
        );
        $log_stmt->execute();
        
        $conn->commit();
        $_SESSION['success_message'] = "Ticket successfully reissued with reference: " . $new_ticket_ref;
        header('Location: all_new_tickets.php');
        exit();
        
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = $e->getMessage();
    }
}
?>

<!doctype html>
<html lang="en" data-layout="vertical" data-topbar="light" data-sidebar="dark" data-sidebar-size="lg" data-sidebar-image="none" data-preloader="disable">

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Reissue Ticket')); ?>
    <?php include 'layouts/head-css.php'; ?>
    
    <!-- SweetAlert2 CSS -->
    <link href="assets/libs/sweetalert2/sweetalert2.min.css" rel="stylesheet" type="text/css" />
    
    <style>
        .card-header {
            background-color: #f8f9fa;
        }
        .form-control:disabled {
            background-color: #e9ecef;
        }
    </style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Ticket Management', 'title' => 'Reissue Ticket')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Reissue Cancelled Ticket</h4>
                                </div>
                                
                                <div class="card-body">
                                    <form action="" method="POST" id="reissueForm">
                                        <div class="row">
                                            <!-- Original Ticket Information (Read-only) -->
                                            <div class="col-md-6">
                                                <h5 class="text-muted">Original Ticket Information</h5>
                                                <hr>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Original Ticket Reference</label>
                                                    <input type="text" class="form-control" value="<?php echo htmlspecialchars($ticket['ticket_ref']); ?>" disabled>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Customer Name</label>
                                                    <input type="text" class="form-control" value="<?php echo htmlspecialchars($ticket['salutation'] . ' ' . $ticket['customer_name']); ?>" disabled>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Original Route</label>
                                                    <input type="text" class="form-control" value="<?php echo htmlspecialchars($ticket['flight_route']); ?>" disabled>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Original Airline</label>
                                                    <input type="text" class="form-control" value="<?php echo htmlspecialchars($ticket['airline']); ?>" disabled>
                                                </div>
                                            </div>
                                            
                                            <!-- New Ticket Information (Editable) -->
                                            <div class="col-md-6">
                                                <h5 class="text-primary">New Ticket Information</h5>
                                                <hr>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">New Ticket Reference <span class="text-danger">*</span></label>
                                                    <input type="text" class="form-control" name="ticket_ref" required value="<?php echo htmlspecialchars($ticket['ticket_ref']); ?>" autocomplete="off">
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Booking Class <span class="text-danger">*</span></label>
                                                    <select class="form-select" name="booking_class" required>
                                                        <option value="Y" <?php echo $ticket['booking_class'] == 'Y' ? 'selected' : ''; ?>>Y - Economy</option>
                                                        <option value="L" <?php echo $ticket['booking_class'] == 'L' ? 'selected' : ''; ?>>L - Premium Economy</option>
                                                        <option value="A" <?php echo $ticket['booking_class'] == 'A' ? 'selected' : ''; ?>>A - Business</option>
                                                    </select>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Airline <span class="text-danger">*</span></label>
                                                    <input type="text" class="form-control" name="airline" required value="<?php echo htmlspecialchars($ticket['airline']); ?>">
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Flight Route <span class="text-danger">*</span></label>
                                                    <input type="text" class="form-control" name="flight_route" required value="<?php echo htmlspecialchars($ticket['flight_route']); ?>">
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Flight Date & Time <span class="text-danger">*</span></label>
                                                    <input type="datetime-local" class="form-control" name="flight_date" required value="<?php echo date('Y-m-d\TH:i', strtotime($ticket['flight_date'])); ?>">
                                                </div>
                                                
                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">Ticket Cost <span class="text-danger">*</span></label>
                                                            <input type="number" class="form-control" name="ticket_cost" step="0.01" required value="<?php echo $ticket['ticket_cost']; ?>">
                                                        </div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <div class="mb-3">
                                                            <label class="form-label">Agency Fee <span class="text-danger">*</span></label>
                                                            <input type="number" class="form-control" name="agency_fee" step="0.01" required value="<?php echo $ticket['agency_fee']; ?>">
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Currency <span class="text-danger">*</span></label>
                                                    <select class="form-select" name="currency" required>
                                                        <option value="USD" <?php echo $ticket['currency'] == 'USD' ? 'selected' : ''; ?>>USD</option>
                                                        <option value="KES" <?php echo $ticket['currency'] == 'KES' ? 'selected' : ''; ?>>KES</option>
                                                    </select>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Notes (Optional)</label>
                                                    <textarea class="form-control" name="notes" rows="3" placeholder="Any additional notes for reissuing this ticket..."></textarea>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="row mt-4">
                                            <div class="col-12">
                                                <div class="d-flex justify-content-end gap-2">
                                                    <a href="all_cancellations.php" class="btn btn-light btn-label waves-effect waves-light rounded-pill">
                                                        <i class="ri-arrow-left-line label-icon align-middle rounded-pill fs-16 me-2"></i> Cancel
                                                    </a>
                                                    <button type="submit" class="btn btn-primary btn-label waves-effect waves-light rounded-pill">
                                                        <i class="ri-save-line label-icon align-middle rounded-pill fs-16 me-2"></i> Reissue Ticket
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    
    <!-- SweetAlert2 JS -->
    <script src="assets/libs/sweetalert2/sweetalert2.min.js"></script>
    <script src="assets/js/app.js"></script>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Form submission with validation
        document.getElementById('reissueForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            Swal.fire({
                title: 'Confirm Reissue',
                text: 'Are you sure you want to reissue this ticket?',
                icon: 'question',
                showCancelButton: true,
                confirmButtonColor: '#3085d6',
                cancelButtonColor: '#d33',
                confirmButtonText: 'Yes, reissue it!',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    this.submit();
                }
            });
        });
        
        // Show success/error messages
        <?php if (isset($_SESSION['success_message'])): ?>
        Swal.fire({
            title: 'Success!',
            text: '<?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>',
            icon: 'success',
            confirmButtonText: 'OK'
        });
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error_message'])): ?>
        Swal.fire({
            title: 'Error!',
            text: '<?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>',
            icon: 'error',
            confirmButtonText: 'OK'
        });
        <?php endif; ?>
        
        // Auto-calculate total amount
        const ticketCost = document.querySelector('input[name="ticket_cost"]');
        const agencyFee = document.querySelector('input[name="agency_fee"]');
        
        function updateTotal() {
            const cost = parseFloat(ticketCost.value) || 0;
            const fee = parseFloat(agencyFee.value) || 0;
            const total = cost + fee;
            
            // Create or update total display
            let totalDisplay = document.getElementById('totalAmount');
            if (!totalDisplay) {
                totalDisplay = document.createElement('div');
                totalDisplay.id = 'totalAmount';
                totalDisplay.className = 'alert alert-info mt-3';
                agencyFee.parentNode.parentNode.parentNode.appendChild(totalDisplay);
            }
            totalDisplay.textContent = `Total Amount: ${total.toFixed(2)}`;
        }
        
        ticketCost.addEventListener('input', updateTotal);
        agencyFee.addEventListener('input', updateTotal);
        updateTotal(); // Initial calculation
    });
    </script>
</body>
</html>