<?php
require_once __DIR__ . '/vendor/phpoffice/vendor/autoload.php';
include 'layouts/session.php';
include 'layouts/dbconn.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');
$currency_filter = isset($_GET['currency']) ? $_GET['currency'] : 'all';

$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

$sheet->setCellValue('A1', 'Visa Applications Report');
$sheet->mergeCells('A1:I1');
$sheet->getStyle('A1:I1')->getFont()->setBold(true)->setSize(14);
$sheet->getStyle('A1:I1')->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);

$sheet->setCellValue('A2', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
$sheet->mergeCells('A2:I2');

$sheet->setCellValue('A4', 'Reference Number');
$sheet->setCellValue('B4', 'Date');
$sheet->setCellValue('C4', 'Client Name');
$sheet->setCellValue('D4', 'Care Of');
$sheet->setCellValue('E4', 'Amount');
$sheet->setCellValue('F4', 'Payment');
$sheet->setCellValue('G4', 'Commission');
$sheet->setCellValue('H4', 'Status');
$sheet->setCellValue('I4', 'Processed By');

$sheet->getStyle('A4:I4')->getFont()->setBold(true);
$sheet->getStyle('A4:I4')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');

$sql = "SELECT va.*, u_created.full_name as created_by_name
        FROM visa_applications va
        LEFT JOIN users u_created ON va.created_by = u_created.user_id
        WHERE va.created_at BETWEEN ? AND ?";

if ($currency_filter !== 'all') {
    $sql .= " AND va.currency = ?";
}

$stmt = $conn->prepare($sql);
if ($currency_filter !== 'all') {
    $stmt->bind_param("sss", $start_date, $end_date, $currency_filter);
} else {
    $stmt->bind_param("ss", $start_date, $end_date);
}

$stmt->execute();
$result = $stmt->get_result();

$row = 5;
$totals = [];

while ($visa = $result->fetch_assoc()) {
    $sheet->setCellValue('A' . $row, $visa['reference_number']);
    $sheet->setCellValue('B' . $row, date('d M Y', strtotime($visa['created_at'])));
    $sheet->setCellValue('C' . $row, $visa['client_name']);
    $sheet->setCellValue('D' . $row, $visa['care_of'] ?? '-');
    $sheet->setCellValue('E' . $row, $visa['currency'] . ' ' . number_format($visa['amount'], 2));
    $sheet->setCellValue('F' . $row, $visa['currency'] . ' ' . number_format($visa['payment'], 2));
    $sheet->setCellValue('G' . $row, $visa['currency'] . ' ' . number_format($visa['commission'], 2));
    $sheet->setCellValue('H' . $row, ucfirst($visa['status']));
    $sheet->setCellValue('I' . $row, $visa['created_by_name']);

    if (!isset($totals[$visa['currency']])) {
        $totals[$visa['currency']] = [
            'amount' => 0,
            'payment' => 0,
            'commission' => 0
        ];
    }
    $totals[$visa['currency']]['amount'] += $visa['amount'];
    $totals[$visa['currency']]['payment'] += $visa['payment'];
    $totals[$visa['currency']]['commission'] += $visa['commission'];

    $row++;
}

$row += 2;

foreach ($totals as $currency => $total) {
    $sheet->setCellValue('A' . $row, 'Totals for ' . $currency);
    $sheet->mergeCells('A' . $row . ':D' . $row);
    $sheet->setCellValue('E' . $row, $currency . ' ' . number_format($total['amount'], 2));
    $sheet->setCellValue('F' . $row, $currency . ' ' . number_format($total['payment'], 2));
    $sheet->setCellValue('G' . $row, $currency . ' ' . number_format($total['commission'], 2));
    $sheet->getStyle('A' . $row . ':I' . $row)->getFont()->setBold(true);
    $row++;
}

foreach(range('A','I') as $col) {
    $sheet->getColumnDimension($col)->setAutoSize(true);
}

$sheet->getStyle('A4:I'.($row-1))->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);

$writer = new Xlsx($spreadsheet);

header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="visa_report_' . date('Y-m-d') . '.xlsx"');
header('Cache-Control: max-age=0');

$writer->save('php://output');
exit;
?>