<?php
include 'layouts/session.php';
include 'layouts/main.php';
include 'layouts/dbconn.php';

$report_type = isset($_GET['report_type']) ? $_GET['report_type'] : 'manifest';
$date_from = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$date_to = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');

function getCargoReport($conn, $report_type, $date_from, $date_to) {
    switch($report_type) {
        case 'manifest':
            return $conn->query("
                SELECT 
                    cb.booking_number,
                    c.first_name as sender_name,
                    c.last_name as sender_lastname,
                    cb.receiver_name,
                    cb.cargo_type,
                    cb.weight,
                    cb.dimensions,
                    cb.booking_status,
                    cb.estimated_delivery_date,
                    cb.total_amount,
                    a1.street_address as pickup_address,
                    a2.street_address as delivery_address
                FROM cargo_bookings cb
                JOIN customers c ON cb.sender_customer_id = c.customer_id
                JOIN customer_addresses a1 ON cb.sender_address_id = a1.address_id
                JOIN customer_addresses a2 ON cb.receiver_address_id = a2.address_id
                WHERE cb.booking_date BETWEEN '$date_from' AND '$date_to'
                ORDER BY cb.booking_date DESC"
            );

        case 'delivery_status':
            return $conn->query("
                SELECT 
                    cb.booking_number,
                    ds.status_type,
                    ds.location,
                    ds.status_date,
                    ds.notes,
                    u.full_name as updated_by
                FROM delivery_status ds
                JOIN cargo_bookings cb ON ds.booking_id = cb.booking_id
                LEFT JOIN users u ON ds.updated_by = u.user_id
                WHERE ds.status_date BETWEEN '$date_from' AND '$date_to'
                ORDER BY ds.status_date DESC"
            );

        case 'issues':
            return $conn->query("
                SELECT 
                    cb.booking_number,
                    i.issue_type,
                    i.description,
                    i.priority,
                    i.status,
                    u1.full_name as reported_by,
                    u2.full_name as assigned_to,
                    i.resolution,
                    i.resolved_date
                FROM issues i
                JOIN cargo_bookings cb ON i.booking_id = cb.booking_id
                JOIN users u1 ON i.reported_by = u1.user_id
                LEFT JOIN users u2 ON i.assigned_to = u2.user_id
                WHERE i.created_at BETWEEN '$date_from' AND '$date_to'
                ORDER BY i.created_at DESC"
            );

        case 'performance':
            return $conn->query("
                SELECT 
                    DATE_FORMAT(booking_date, '%Y-%m') as month,
                    COUNT(*) as total_bookings,
                    COUNT(CASE WHEN booking_status = 'delivered' THEN 1 END) as delivered,
                    COUNT(CASE WHEN booking_status = 'failed_delivery' THEN 1 END) as failed,
                    AVG(CASE 
                        WHEN booking_status = 'delivered' 
                        THEN DATEDIFF(actual_delivery_date, estimated_delivery_date) 
                    END) as avg_delivery_delay,
                    SUM(total_amount) as revenue
                FROM cargo_bookings
                WHERE booking_date BETWEEN '$date_from' AND '$date_to'
                GROUP BY DATE_FORMAT(booking_date, '%Y-%m')
                ORDER BY month DESC"
            );
    }
}

$result = getCargoReport($conn, $report_type, $date_from, $date_to);
?>

<!doctype html>
<html lang="en" data-layout="vertical" data-topbar="light" data-sidebar="dark" data-sidebar-size="lg">
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Cargo Reports')); ?>
    <?php include 'layouts/head-css.php'; ?>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Reports', 'title' => 'Cargo Reports')); ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header d-flex justify-content-between align-items-center">
                                    <h4 class="card-title mb-0">Cargo Reports</h4>
                                    <div>
                                        <button onclick="printReport()" class="btn btn-info me-2">Print</button>
                                        <button onclick="exportToCSV()" class="btn btn-success">Export to CSV</button>
                                    </div>
                                </div>
                                <div class="card-body">
                                    <form class="row g-3 mb-4" method="GET">
                                        <div class="col-md-3">
                                            <select class="form-select" name="report_type">
                                                <option value="manifest" <?php echo $report_type == 'manifest' ? 'selected' : ''; ?>>Cargo Manifest</option>
                                                <option value="delivery_status" <?php echo $report_type == 'delivery_status' ? 'selected' : ''; ?>>Delivery Status</option>
                                                <option value="issues" <?php echo $report_type == 'issues' ? 'selected' : ''; ?>>Issues Report</option>
                                                <option value="performance" <?php echo $report_type == 'performance' ? 'selected' : ''; ?>>Performance Metrics</option>
                                            </select>
                                        </div>
                                        <div class="col-md-3">
                                            <input type="date" class="form-control" name="date_from" 
                                                   value="<?php echo htmlspecialchars($date_from); ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <input type="date" class="form-control" name="date_to" 
                                                   value="<?php echo htmlspecialchars($date_to); ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <button type="submit" class="btn btn-primary w-100">Generate Report</button>
                                        </div>
                                    </form>

                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <?php if ($report_type == 'manifest'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Booking #</th>
                                                        <th>Sender</th>
                                                        <th>Receiver</th>
                                                        <th>Type</th>
                                                        <th>Weight</th>
                                                        <th>Dimensions</th>
                                                        <th>Status</th>
                                                        <th>Est. Delivery</th>
                                                        <th>Amount</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['booking_number']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['sender_name'] . ' ' . $row['sender_lastname']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['receiver_name']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['cargo_type']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['weight']); ?> kg</td>
                                                            <td><?php echo htmlspecialchars($row['dimensions']); ?></td>
                                                            <td>
                                                                <span class="badge bg-<?php 
                                                                    echo $row['booking_status'] == 'delivered' ? 'success' : 
                                                                        ($row['booking_status'] == 'in_transit' ? 'info' : 
                                                                        ($row['booking_status'] == 'failed_delivery' ? 'danger' : 'warning')); 
                                                                ?>">
                                                                    <?php echo ucfirst(str_replace('_', ' ', $row['booking_status'])); ?>
                                                                </span>
                                                            </td>
                                                            <td><?php echo date('d M Y', strtotime($row['estimated_delivery_date'])); ?></td>
                                                            <td><?php echo number_format($row['total_amount'], 2); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'delivery_status'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Booking #</th>
                                                        <th>Status</th>
                                                        <th>Location</th>
                                                        <th>Date</th>
                                                        <th>Notes</th>
                                                        <th>Updated By</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['booking_number']); ?></td>
                                                            <td><?php echo ucfirst(str_replace('_', ' ', $row['status_type'])); ?></td>
                                                            <td><?php echo htmlspecialchars($row['location']); ?></td>
                                                            <td><?php echo date('d M Y H:i', strtotime($row['status_date'])); ?></td>
                                                            <td><?php echo htmlspecialchars($row['notes']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['updated_by']); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'issues'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Booking #</th>
                                                        <th>Issue Type</th>
                                                        <th>Description</th>
                                                        <th>Priority</th>
                                                        <th>Status</th>
                                                        <th>Reported By</th>
                                                        <th>Assigned To</th>
                                                        <th>Resolution</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo htmlspecialchars($row['booking_number']); ?></td>
                                                            <td><?php echo ucfirst($row['issue_type']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['description']); ?></td>
                                                            <td>
                                                                <span class="badge bg-<?php 
                                                                    echo $row['priority'] == 'critical' ? 'danger' : 
                                                                        ($row['priority'] == 'high' ? 'warning' : 
                                                                        ($row['priority'] == 'medium' ? 'info' : 'success')); 
                                                                ?>">
                                                                    <?php echo ucfirst($row['priority']); ?>
                                                                </span>
                                                            </td>
                                                            <td><?php echo ucfirst($row['status']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['reported_by']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['assigned_to'] ?? 'Unassigned'); ?></td>
                                                            <td><?php echo htmlspecialchars($row['resolution'] ?? 'Pending'); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>

                                            <?php elseif ($report_type == 'performance'): ?>
                                                <thead>
                                                    <tr>
                                                        <th>Month</th>
                                                        <th>Total Bookings</th>
                                                        <th>Delivered</th>
                                                        <th>Failed</th>
                                                        <th>Success Rate</th>
                                                        <th>Avg Delay (Days)</th>
                                                        <th>Revenue</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php while ($row = $result->fetch_assoc()): ?>
                                                        <tr>
                                                            <td><?php echo date('M Y', strtotime($row['month'] . '-01')); ?></td>
                                                            <td><?php echo htmlspecialchars($row['total_bookings']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['delivered']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['failed']); ?></td>
                                                            <td><?php echo number_format(($row['delivered'] / $row['total_bookings']) * 100, 1) . '%'; ?></td>
                                                            <td><?php echo round($row['avg_delivery_delay'], 1); ?></td>
                                                            <td><?php echo number_format($row['revenue'], 2); ?></td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                </tbody>
                                            <?php endif; ?>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>

    <?php include 'layouts/vendor-scripts.php'; ?>
    
    <script>
    function exportToCSV() {
        const urlParams = new URLSearchParams(window.location.search);
        const params = {
            report_type: urlParams.get('report_type') || 'manifest',
            date_from: urlParams.get('date_from') || '',
            date_to: urlParams.get('date_to') || ''
        };
        window.location.href = `export_cargo_report.php?${new URLSearchParams(params)}`;
    }

    function printReport() {
        window.print();
    }
    </script>
</body>
</html>