<?php
include 'layouts/main.php';
include 'layouts/session.php';
include 'layouts/dbconn.php';

// Get filter parameters
$staff_id = isset($_GET['staff_id']) ? $_GET['staff_id'] : '';
$service_type = isset($_GET['service_type']) ? $_GET['service_type'] : 'all';
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Get list of staff
$staff_sql = "SELECT staff_id, full_name FROM staff WHERE status = 'active' ORDER BY full_name";
$staff_result = $conn->query($staff_sql);

// Function to get ticket details by staff (assuming staff_id is added to ticket_details table)
function getTicketDetails($conn, $staff_id, $start_date, $end_date) {
    $sql = "SELECT td.*, s.full_name as staff_name
            FROM ticket_details td
            LEFT JOIN staff s ON td.staff_id = s.staff_id
            WHERE DATE(td.flight_date) BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND td.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY td.flight_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Function to get visa applications by staff
function getVisaApplications($conn, $staff_id, $start_date, $end_date) {
    $sql = "SELECT va.*, s.full_name as staff_name
            FROM visa_applications va
            LEFT JOIN staff s ON va.staff_id = s.staff_id
            WHERE va.created_at BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND va.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY va.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Function to get passport extensions by staff
function getPassportExtensions($conn, $staff_id, $start_date, $end_date) {
    $sql = "SELECT pe.*, s.full_name as staff_name
            FROM passport_extensions pe
            LEFT JOIN staff s ON pe.staff_id = s.staff_id
            WHERE pe.extension_date BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND pe.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY pe.extension_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Function to get go home services by staff
function getGoHomeServices($conn, $staff_id, $start_date, $end_date) {
    $sql = "SELECT ghs.*, s.full_name as staff_name
            FROM go_home_service ghs
            LEFT JOIN staff s ON ghs.staff_id = s.staff_id
            WHERE ghs.service_date BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND ghs.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY ghs.service_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Get selected staff's name if applicable
$selected_staff_name = '';
if (!empty($staff_id)) {
    $staff_name_sql = "SELECT full_name FROM staff WHERE staff_id = ?";
    $stmt = $conn->prepare($staff_name_sql);
    $stmt->bind_param("i", $staff_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $selected_staff_name = $row['full_name'];
    }
}

// Get data based on service type
$tickets = $visas = $extensions = $goHomeServices = null;

if ($service_type == 'all' || $service_type == 'tickets') {
    $tickets = getTicketDetails($conn, $staff_id, $start_date, $end_date);
}
if ($service_type == 'all' || $service_type == 'visas') {
    $visas = getVisaApplications($conn, $staff_id, $start_date, $end_date);
}
if ($service_type == 'all' || $service_type == 'extensions') {
    $extensions = getPassportExtensions($conn, $staff_id, $start_date, $end_date);
}
if ($service_type == 'all' || $service_type == 'go_home') {
    $goHomeServices = getGoHomeServices($conn, $staff_id, $start_date, $end_date);
}
?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Staff Services Report')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <style>
        .service-section {
            margin-bottom: 30px;
        }
        .section-header {
            background-color: #f8f9fa;
            padding: 10px 15px;
            margin-top: 20px;
            margin-bottom: 10px;
            border-radius: 4px;
        }
        @media print {
            .no-print {
                display: none !important;
            }
            .card {
                border: 1px solid #dee2e6 !important;
            }
        }
        .summary-card {
            margin-bottom: 20px;
        }
    </style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Reports', 'title' => 'Staff Services Report')); ?>
                    
                    <!-- Filter Section -->
                    <div class="row mb-3 no-print">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-body">
                                    <form method="GET" class="row g-3">
                                        <div class="col-md-3">
                                            <label class="form-label">Staff Member</label>
                                            <select name="staff_id" class="form-select">
                                                <option value="">All Staff</option>
                                                <?php while ($staff = $staff_result->fetch_assoc()) { ?>
                                                    <option value="<?php echo $staff['staff_id']; ?>" 
                                                            <?php echo ($staff_id == $staff['staff_id']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($staff['full_name']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                        <div class="col-md-2">
                                            <label class="form-label">Service Type</label>
                                            <select name="service_type" class="form-select">
                                                <option value="all" <?php echo ($service_type == 'all') ? 'selected' : ''; ?>>All Services</option>
                                                <option value="tickets" <?php echo ($service_type == 'tickets') ? 'selected' : ''; ?>>Tickets</option>
                                                <option value="visas" <?php echo ($service_type == 'visas') ? 'selected' : ''; ?>>Visas</option>
                                                <option value="extensions" <?php echo ($service_type == 'extensions') ? 'selected' : ''; ?>>Extensions</option>
                                                <option value="go_home" <?php echo ($service_type == 'go_home') ? 'selected' : ''; ?>>Go Home</option>
                                            </select>
                                        </div>
                                        <div class="col-md-2">
                                            <label class="form-label">Start Date</label>
                                            <input type="date" class="form-control" name="start_date" value="<?php echo $start_date; ?>">
                                        </div>
                                        <div class="col-md-2">
                                            <label class="form-label">End Date</label>
                                            <input type="date" class="form-control" name="end_date" value="<?php echo $end_date; ?>">
                                        </div>
                                        <div class="col-md-3">
                                            <label class="form-label">&nbsp;</label>
                                            <div>
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="ri-search-line"></i> Generate Report
                                                </button>
                                                <button type="button" onclick="window.print()" class="btn btn-secondary">
                                                    <i class="ri-printer-line"></i> Print
                                                </button>
                                                <button type="button" onclick="exportToExcel()" class="btn btn-success">
                                                    <i class="ri-file-excel-line"></i> Export
                                                </button>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Report Header -->
                    <div class="row">
                        <div class="col-md-12">
                            <h5 class="mb-3">
                                <?php if (!empty($staff_id)) { ?>
                                    Staff: <?php echo htmlspecialchars($selected_staff_name); ?> |
                                <?php } ?>
                                Period: <?php echo date('d M Y', strtotime($start_date)); ?> to <?php echo date('d M Y', strtotime($end_date)); ?>
                            </h5>
                        </div>
                    </div>
                    
                    <!-- Ticket Sales Section -->
                    <?php if ($service_type == 'all' || $service_type == 'tickets') { ?>
                    <div class="service-section">
                        <div class="section-header">
                            <h4 class="mb-0">Ticket Sales</h4>
                        </div>
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <?php if (empty($staff_id)) { ?><th>Staff</th><?php } ?>
                                                <th>Customer</th>
                                                <th>Ticket Reference</th>
                                                <th>Airline</th>
                                                <th>Route</th>
                                                <th>Ticket Cost</th>
                                                <th>Agency Fee</th>
                                                <th>Total</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $total_ticket_cost = 0;
                                            $total_ticket_commission = 0;
                                            
                                            if ($tickets && $tickets->num_rows > 0) {
                                                while ($ticket = $tickets->fetch_assoc()) {
                                                    $total_ticket_cost += $ticket['ticket_cost'];
                                                    $total_ticket_commission += $ticket['agency_fee'];
                                            ?>
                                            <tr>
                                                <td><?php echo date('d M Y', strtotime($ticket['flight_date'])); ?></td>
                                                <?php if (empty($staff_id)) { ?>
                                                    <td><?php echo htmlspecialchars($ticket['staff_name'] ?? 'N/A'); ?></td>
                                                <?php } ?>
                                                <td><?php echo htmlspecialchars($ticket['customer_name']); ?></td>
                                                <td><?php echo htmlspecialchars($ticket['ticket_ref']); ?></td>
                                                <td><?php echo htmlspecialchars($ticket['airline']); ?></td>
                                                <td><?php echo htmlspecialchars($ticket['flight_route']); ?></td>
                                                <td><?php echo $ticket['currency']; ?> <?php echo number_format($ticket['ticket_cost'], 2); ?></td>
                                                <td><?php echo $ticket['currency']; ?> <?php echo number_format($ticket['agency_fee'], 2); ?></td>
                                                <td><?php echo $ticket['currency']; ?> <?php echo number_format($ticket['total_amount'], 2); ?></td>
                                            </tr>
                                            <?php 
                                                }
                                            } else {
                                            ?>
                                            <tr>
                                                <td colspan="<?php echo empty($staff_id) ? '9' : '8'; ?>" class="text-center">No ticket records found</td>
                                            </tr>
                                            <?php } ?>
                                        </tbody>
                                        <tfoot>
                                            <tr class="table-active">
                                                <th colspan="<?php echo empty($staff_id) ? '6' : '5'; ?>">Total</th>
                                                <th><?php echo number_format($total_ticket_cost, 2); ?></th>
                                                <th><?php echo number_format($total_ticket_commission, 2); ?></th>
                                                <th><?php echo number_format($total_ticket_cost + $total_ticket_commission, 2); ?></th>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php } ?>
                    
                    <!-- Visa Applications Section -->
                    <?php if ($service_type == 'all' || $service_type == 'visas') { ?>
                    <div class="service-section">
                        <div class="section-header">
                            <h4 class="mb-0">Visa Applications</h4>
                        </div>
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <?php if (empty($staff_id)) { ?><th>Staff</th><?php } ?>
                                                <th>Client Name</th>
                                                <th>Reference Number</th>
                                                <th>Amount</th>
                                                <th>Payment</th>
                                                <th>Commission</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $total_visa_amount = 0;
                                            $total_visa_commission = 0;
                                            
                                            if ($visas && $visas->num_rows > 0) {
                                                while ($visa = $visas->fetch_assoc()) {
                                                    $total_visa_amount += $visa['amount'];
                                                    $total_visa_commission += $visa['commission'];
                                            ?>
                                            <tr>
                                                <td><?php echo date('d M Y', strtotime($visa['created_at'])); ?></td>
                                                <?php if (empty($staff_id)) { ?>
                                                    <td><?php echo htmlspecialchars($visa['staff_name'] ?? 'N/A'); ?></td>
                                                <?php } ?>
                                                <td><?php echo htmlspecialchars($visa['client_name']); ?></td>
                                                <td><?php echo htmlspecialchars($visa['reference_number']); ?></td>
                                                <td><?php echo $visa['currency']; ?> <?php echo number_format($visa['amount'], 2); ?></td>
                                                <td><?php echo $visa['currency']; ?> <?php echo number_format($visa['payment'], 2); ?></td>
                                                <td><?php echo $visa['currency']; ?> <?php echo number_format($visa['commission'], 2); ?></td>
                                                <td><?php echo ucfirst($visa['status']); ?></td>
                                            </tr>
                                            <?php 
                                                }
                                            } else {
                                            ?>
                                            <tr>
                                                <td colspan="<?php echo empty($staff_id) ? '8' : '7'; ?>" class="text-center">No visa records found</td>
                                            </tr>
                                            <?php } ?>
                                        </tbody>
                                        <tfoot>
                                            <tr class="table-active">
                                                <th colspan="<?php echo empty($staff_id) ? '4' : '3'; ?>">Total</th>
                                                <th><?php echo number_format($total_visa_amount, 2); ?></th>
                                                <th colspan="1"></th>
                                                <th><?php echo number_format($total_visa_commission, 2); ?></th>
                                                <th></th>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php } ?>
                    
                    <!-- Passport Extensions Section -->
                    <?php if ($service_type == 'all' || $service_type == 'extensions') { ?>
                    <div class="service-section">
                        <div class="section-header">
                            <h4 class="mb-0">Passport Extensions</h4>
                        </div>
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <?php if (empty($staff_id)) { ?><th>Staff</th><?php } ?>
                                                <th>Client Name</th>
                                                <th>Passport Number</th>
                                                <th>Type</th>
                                                <th>Amount Received</th>
                                                <th>Service Payment</th>
                                                <th>Agency Fee</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $total_ext_amount = 0;
                                            $total_ext_agency_fee = 0;
                                            
                                            if ($extensions && $extensions->num_rows > 0) {
                                                while ($ext = $extensions->fetch_assoc()) {
                                                    $total_ext_amount += $ext['amount_received'];
                                                    $total_ext_agency_fee += $ext['agency_fee'];
                                            ?>
                                            <tr>
                                                <td><?php echo date('d M Y', strtotime($ext['extension_date'])); ?></td>
                                                <?php if (empty($staff_id)) { ?>
                                                    <td><?php echo htmlspecialchars($ext['staff_name'] ?? 'N/A'); ?></td>
                                                <?php } ?>
                                                <td><?php echo htmlspecialchars($ext['client_name']); ?></td>
                                                <td><?php echo htmlspecialchars($ext['passport_number']); ?></td>
                                                <td><?php echo ucfirst($ext['service_type']); ?></td>
                                                <td><?php echo $ext['currency']; ?> <?php echo number_format($ext['amount_received'], 2); ?></td>
                                                <td><?php echo $ext['currency']; ?> <?php echo number_format($ext['service_payment'], 2); ?></td>
                                                <td><?php echo $ext['currency']; ?> <?php echo number_format($ext['agency_fee'], 2); ?></td>
                                                <td><?php echo ucfirst($ext['payment_status']); ?></td>
                                            </tr>
                                            <?php 
                                                }
                                            } else {
                                            ?>
                                            <tr>
                                                <td colspan="<?php echo empty($staff_id) ? '9' : '8'; ?>" class="text-center">No extension records found</td>
                                            </tr>
                                            <?php } ?>
                                        </tbody>
                                        <tfoot>
                                            <tr class="table-active">
                                                <th colspan="<?php echo empty($staff_id) ? '5' : '4'; ?>">Total</th>
                                                <th><?php echo number_format($total_ext_amount, 2); ?></th>
                                                <th colspan="1"></th>
                                                <th><?php echo number_format($total_ext_agency_fee, 2); ?></th>
                                                <th></th>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php } ?>
                    
                    <!-- Go Home Services Section -->
                    <?php if ($service_type == 'all' || $service_type == 'go_home') { ?>
                    <div class="service-section">
                        <div class="section-header">
                            <h4 class="mb-0">Go Home Services</h4>
                        </div>
                        <div class="card">
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <?php if (empty($staff_id)) { ?><th>Staff</th><?php } ?>
                                                <th>Client Name</th>
                                                <th>Mother Name</th>
                                                <th>DOB</th>
                                                <th>Reference</th>
                                                <th>Price</th>
                                                <th>Office Fee</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $total_go_home_price = 0;
                                            $total_go_home_fee = 0;
                                            
                                            if ($goHomeServices && $goHomeServices->num_rows > 0) {
                                                while ($ghs = $goHomeServices->fetch_assoc()) {
                                                    $total_go_home_price += $ghs['price'];
                                                    $total_go_home_fee += $ghs['office_fee'];
                                            ?>
                                            <tr>
                                                <td><?php echo date('d M Y', strtotime($ghs['service_date'])); ?></td>
                                                <?php if (empty($staff_id)) { ?>
                                                    <td><?php echo htmlspecialchars($ghs['staff_name'] ?? 'N/A'); ?></td>
                                                <?php } ?>
                                                <td><?php echo htmlspecialchars($ghs['client_name']); ?></td>
                                                <td><?php echo htmlspecialchars($ghs['mother_name']); ?></td>
                                                <td><?php echo date('d M Y', strtotime($ghs['date_of_birth'])); ?></td>
                                                <td><?php echo htmlspecialchars($ghs['reference_number']); ?></td>
                                                <td><?php echo $ghs['currency']; ?> <?php echo number_format($ghs['price'], 2); ?></td>
                                                <td><?php echo $ghs['currency']; ?> <?php echo number_format($ghs['office_fee'], 2); ?></td>
                                                <td><?php echo ucfirst($ghs['status']); ?></td>
                                            </tr>
                                            <?php 
                                                }
                                            } else {
                                            ?>
                                            <tr>
                                                <td colspan="<?php echo empty($staff_id) ? '9' : '8'; ?>" class="text-center">No go home service records found</td>
                                            </tr>
                                            <?php } ?>
                                        </tbody>
                                        <tfoot>
                                            <tr class="table-active">
                                                <th colspan="<?php echo empty($staff_id) ? '6' : '5'; ?>">Total</th>
                                                <th><?php echo number_format($total_go_home_price, 2); ?></th>
                                                <th><?php echo number_format($total_go_home_fee, 2); ?></th>
                                                <th></th>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php } ?>
                    
                    <!-- Overall Summary -->
                    <?php if ($service_type == 'all') { ?>
                    <div class="row mt-4">
                        <div class="col-md-12">
                            <div class="card summary-card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Overall Summary</h4>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="text-center">
                                                <h5>Total Revenue</h5>
                                                <h3><?php echo number_format($total_ticket_cost + $total_visa_amount + $total_ext_amount + $total_go_home_price, 2); ?></h3>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="text-center">
                                                <h5>Total Commission</h5>
                                                <h3><?php echo number_format($total_ticket_commission + $total_visa_commission + $total_ext_agency_fee + $total_go_home_fee, 2); ?></h3>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="text-center">
                                                <h5>Total Transactions</h5>
                                                <h3>
                                                    <?php 
                                                    $total_transactions = 0;
                                                    if ($tickets) $total_transactions += $tickets->num_rows;
                                                    if ($visas) $total_transactions += $visas->num_rows;
                                                    if ($extensions) $total_transactions += $extensions->num_rows;
                                                    if ($goHomeServices) $total_transactions += $goHomeServices->num_rows;
                                                    echo $total_transactions;
                                                    ?>
                                                </h3>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php } ?>
                    
                </div>
            </div>
            
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>
    
    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="assets/js/app.js"></script>
    
    <script>
    function exportToExcel() {
        // Get current filter parameters
        const staff_id = '<?php echo $staff_id; ?>';
        const service_type = '<?php echo $service_type; ?>';
        const start_date = '<?php echo $start_date; ?>';
        const end_date = '<?php echo $end_date; ?>';
        
        // Construct the export URL with all parameters
        const exportUrl = `export_staff_services.php?staff_id=${staff_id}&service_type=${service_type}&start_date=${start_date}&end_date=${end_date}`;
        
        // Redirect to export URL
        window.location.href = exportUrl;
    }
    </script>
</body>
</html>