<?php
include 'layouts/main.php';
include 'layouts/session.php';
include 'layouts/dbconn.php';

// Get date range from GET parameters or default to current month
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Function to get agent performance data
function getAgentPerformanceData($conn, $start_date, $end_date) {
    // Overall agent performance summary
    $sql = "
        SELECT 
            a.agent_name,
            a.agent_id,
            COALESCE(t.ticket_count, 0) as ticket_count,
            COALESCE(t.ticket_revenue, 0) as ticket_revenue,
            COALESCE(t.ticket_commission, 0) as ticket_commission,
            COALESCE(v.visa_count, 0) as visa_count,
            COALESCE(v.visa_revenue, 0) as visa_revenue,
            COALESCE(v.visa_commission, 0) as visa_commission,
            COALESCE(e.extension_count, 0) as extension_count,
            COALESCE(e.extension_revenue, 0) as extension_revenue,
            COALESCE(e.extension_commission, 0) as extension_commission,
            COALESCE(g.go_home_count, 0) as go_home_count,
            COALESCE(g.go_home_revenue, 0) as go_home_revenue,
            COALESCE(g.go_home_commission, 0) as go_home_commission,
            -- Calculate totals
            (COALESCE(t.ticket_revenue, 0) + COALESCE(v.visa_revenue, 0) + 
             COALESCE(e.extension_revenue, 0) + COALESCE(g.go_home_revenue, 0)) as total_revenue,
            (COALESCE(t.ticket_commission, 0) + COALESCE(v.visa_commission, 0) + 
             COALESCE(e.extension_commission, 0) + COALESCE(g.go_home_commission, 0)) as total_commission,
            (COALESCE(t.ticket_count, 0) + COALESCE(v.visa_count, 0) + 
             COALESCE(e.extension_count, 0) + COALESCE(g.go_home_count, 0)) as total_transactions
        FROM (
            SELECT user_id as agent_id, full_name as agent_name
            FROM users 
            WHERE role_id IN (1, 2, 3, 5) -- Admin, GM, Operations, Employee
        ) a
        LEFT JOIN (
            SELECT 
                created_by,
                COUNT(*) as ticket_count,
                SUM(ticket_cost) as ticket_revenue,
                SUM(agency_fee) as ticket_commission
            FROM ticket_details
            WHERE DATE(flight_date) BETWEEN ? AND ?
            GROUP BY created_by
        ) t ON a.agent_id = t.created_by
        LEFT JOIN (
            SELECT 
                created_by,
                COUNT(*) as visa_count,
                SUM(amount) as visa_revenue,
                SUM(commission) as visa_commission
            FROM visa_applications
            WHERE created_at BETWEEN ? AND ?
            GROUP BY created_by
        ) v ON a.agent_id = v.created_by
        LEFT JOIN (
            SELECT 
                created_by,
                COUNT(*) as extension_count,
                SUM(amount_received) as extension_revenue,
                SUM(agency_fee) as extension_commission
            FROM passport_extensions
            WHERE extension_date BETWEEN ? AND ?
            GROUP BY created_by
        ) e ON a.agent_id = e.created_by
        LEFT JOIN (
            SELECT 
                agent_name,
                COUNT(*) as go_home_count,
                SUM(price) as go_home_revenue,
                SUM(office_fee) as go_home_commission
            FROM go_home_service
            WHERE service_date BETWEEN ? AND ?
            GROUP BY agent_name
        ) g ON a.agent_name = g.agent_name
        HAVING total_revenue > 0 OR total_commission > 0
        ORDER BY total_revenue DESC
    ";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssssssss", $start_date, $end_date, $start_date, $end_date, 
                      $start_date, $end_date, $start_date, $end_date);
    $stmt->execute();
    return $stmt->get_result();
}

$agentData = getAgentPerformanceData($conn, $start_date, $end_date);
?>

<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Agent Performance Report')); ?>
    <?php include 'layouts/head-css.php'; ?>
    <style>
        .stat-card {
            background: #fff;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #333;
        }
        .stat-label {
            color: #666;
            font-size: 14px;
        }
        .performance-chart {
            max-height: 300px;
            overflow-x: auto;
        }
        .agent-table th {
            background-color: #f8f9fa;
        }
        @media print {
            .no-print {
                display: none !important;
            }
            .card {
                border: 1px solid #dee2e6 !important;
            }
        }
    </style>
</head>

<body>
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>
        
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Reports', 'title' => 'Agent Performance Report')); ?>
                    
                    <!-- Date Range Filter -->
                    <div class="row mb-3 no-print">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-body">
                                    <form method="GET" class="row g-3">
                                        <div class="col-md-4">
                                            <label class="form-label">Start Date</label>
                                            <input type="date" class="form-control" name="start_date" value="<?php echo $start_date; ?>">
                                        </div>
                                        <div class="col-md-4">
                                            <label class="form-label">End Date</label>
                                            <input type="date" class="form-control" name="end_date" value="<?php echo $end_date; ?>">
                                        </div>
                                        <div class="col-md-4">
                                            <label class="form-label">&nbsp;</label>
                                            <div>
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="ri-search-line"></i> Generate Report
                                                </button>
                                                <button type="button" onclick="window.print()" class="btn btn-secondary">
                                                    <i class="ri-printer-line"></i> Print
                                                </button>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Summary Cards -->
                    <div class="row">
                        <div class="col-md-12">
                            <h5 class="mb-3">Period: <?php echo date('d M Y', strtotime($start_date)); ?> to <?php echo date('d M Y', strtotime($end_date)); ?></h5>
                        </div>
                    </div>
                    
                    <!-- Agent Performance Table -->
                    <div class="row">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Agent Performance Overview</h4>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table table-striped table-hover agent-table">
                                            <thead>
                                                <tr>
                                                    <th>Agent Name</th>
                                                    <th>Total Revenue</th>
                                                    <th>Total Commission</th>
                                                    <th>Transactions</th>
                                                    <th>Tickets</th>
                                                    <th>Visas</th>
                                                    <th>Extensions</th>
                                                    <th>Go Home</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php
                                                $total_revenue = 0;
                                                $total_commission = 0;
                                                $total_transactions = 0;
                                                
                                                while ($agent = $agentData->fetch_assoc()) {
                                                    $total_revenue += $agent['total_revenue'];
                                                    $total_commission += $agent['total_commission'];
                                                    $total_transactions += $agent['total_transactions'];
                                                ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($agent['agent_name']); ?></td>
                                                    <td>$<?php echo number_format($agent['total_revenue'], 2); ?></td>
                                                    <td>$<?php echo number_format($agent['total_commission'], 2); ?></td>
                                                    <td><?php echo $agent['total_transactions']; ?></td>
                                                    <td><?php echo $agent['ticket_count']; ?></td>
                                                    <td><?php echo $agent['visa_count']; ?></td>
                                                    <td><?php echo $agent['extension_count']; ?></td>
                                                    <td><?php echo $agent['go_home_count']; ?></td>
                                                </tr>
                                                <?php } ?>
                                            </tbody>
                                            <tfoot>
                                                <tr class="table-active">
                                                    <th>Total</th>
                                                    <th>$<?php echo number_format($total_revenue, 2); ?></th>
                                                    <th>$<?php echo number_format($total_commission, 2); ?></th>
                                                    <th><?php echo $total_transactions; ?></th>
                                                    <th colspan="4"></th>
                                                </tr>
                                            </tfoot>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Service Type Breakdown -->
                    <div class="row mt-4">
                        <!-- Reset mysqli pointer for second read -->
                        <?php
                        $agentData = getAgentPerformanceData($conn, $start_date, $end_date);
                        ?>
                        
                        <!-- Tickets -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Ticket Sales Performance</h4>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table">
                                            <thead>
                                                <tr>
                                                    <th>Agent</th>
                                                    <th>Tickets</th>
                                                    <th>Revenue</th>
                                                    <th>Commission</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php 
                                                $agentData->data_seek(0);
                                                while ($agent = $agentData->fetch_assoc()) { 
                                                    if ($agent['ticket_count'] > 0) {
                                                ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($agent['agent_name']); ?></td>
                                                    <td><?php echo $agent['ticket_count']; ?></td>
                                                    <td>$<?php echo number_format($agent['ticket_revenue'], 2); ?></td>
                                                    <td>$<?php echo number_format($agent['ticket_commission'], 2); ?></td>
                                                </tr>
                                                <?php 
                                                    }
                                                } 
                                                ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Visa Applications -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Visa Applications Performance</h4>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table">
                                            <thead>
                                                <tr>
                                                    <th>Agent</th>
                                                    <th>Applications</th>
                                                    <th>Revenue</th>
                                                    <th>Commission</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php 
                                                $agentData->data_seek(0);
                                                while ($agent = $agentData->fetch_assoc()) { 
                                                    if ($agent['visa_count'] > 0) {
                                                ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($agent['agent_name']); ?></td>
                                                    <td><?php echo $agent['visa_count']; ?></td>
                                                    <td>$<?php echo number_format($agent['visa_revenue'], 2); ?></td>
                                                    <td>$<?php echo number_format($agent['visa_commission'], 2); ?></td>
                                                </tr>
                                                <?php 
                                                    }
                                                } 
                                                ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mt-4">
                        <!-- Passport Extensions -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Passport Extensions Performance</h4>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table">
                                            <thead>
                                                <tr>
                                                    <th>Agent</th>
                                                    <th>Extensions</th>
                                                    <th>Revenue</th>
                                                    <th>Commission</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php 
                                                $agentData->data_seek(0);
                                                while ($agent = $agentData->fetch_assoc()) { 
                                                    if ($agent['extension_count'] > 0) {
                                                ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($agent['agent_name']); ?></td>
                                                    <td><?php echo $agent['extension_count']; ?></td>
                                                    <td>$<?php echo number_format($agent['extension_revenue'], 2); ?></td>
                                                    <td>$<?php echo number_format($agent['extension_commission'], 2); ?></td>
                                                </tr>
                                                <?php 
                                                    }
                                                } 
                                                ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Go Home Service -->
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">Go Home Service Performance</h4>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table">
                                            <thead>
                                                <tr>
                                                    <th>Agent</th>
                                                    <th>Services</th>
                                                    <th>Revenue</th>
                                                    <th>Commission</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php 
                                                $agentData->data_seek(0);
                                                while ($agent = $agentData->fetch_assoc()) { 
                                                    if ($agent['go_home_count'] > 0) {
                                                ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($agent['agent_name']); ?></td>
                                                    <td><?php echo $agent['go_home_count']; ?></td>
                                                    <td>$<?php echo number_format($agent['go_home_revenue'], 2); ?></td>
                                                    <td>$<?php echo number_format($agent['go_home_commission'], 2); ?></td>
                                                </tr>
                                                <?php 
                                                    }
                                                } 
                                                ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                </div>
            </div>
            
            <?php include 'layouts/footer.php'; ?>
        </div>
    </div>
    
    <?php include 'layouts/vendor-scripts.php'; ?>
    <script src="assets/js/app.js"></script>
</body>
</html>