<?php
session_start();
include 'layouts/dbconn.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get user details for permission checks
$user_id = $_SESSION['user_id'];
$sql_user_details = "SELECT username, user_role, full_name, email, role_id FROM users WHERE user_id = $user_id";
$result_user_details = $conn->query($sql_user_details);

if ($result_user_details && $row_user_details = $result_user_details->fetch_assoc()) {
    $username = $row_user_details['username'];
    $user_role = $row_user_details['user_role'];
    $full_name = $row_user_details['full_name'];
    $email = $row_user_details['email'];
    $role_id = $row_user_details['role_id'];
}

// Function to check user permissions
function hasPermission($role_id, $permission) {
    // Define permissions for different roles
    $permissions = [
        '1' => ['view', 'edit', 'delete', 'approve', 'invoice'], // Admin has all permissions
        '2' => ['view', 'edit', 'approve', 'invoice'], // General Manager
        '3' => ['view', 'edit', 'invoice'], // Operations
        '4' => ['view', 'invoice'], // Finance
        '5' => ['view', 'edit'], // Declaration Officer
        '6' => ['view'], // Shipping Line Clerk
        '7' => ['view'], // Transport Supervisor
    ];
    
    if (isset($permissions[$role_id]) && in_array($permission, $permissions[$role_id])) {
        return true;
    }
    
    return false;
}

// Check if user has permission to edit all-in amounts
$can_edit = hasPermission($role_id, 'edit') || hasPermission($role_id, 'approve');

if (!$can_edit) {
    $_SESSION['error_message'] = "You don't have permission to update all-in amounts.";
    header('Location: all_in_list.php');
    exit();
}

// Format date helper function
function formatDate($date) {
    if (empty($date) || $date == '0000-00-00' || $date == '0000-00-00 00:00:00') {
        return 'N/A';
    }
    return date('Y-m-d', strtotime($date));
}

// Initialize variables
$file_number = '';
$file_details = null;
$existing_all_in = 0;
$non_op_costs = 0;
$currency = '$';
$all_non_op_costs = [];
$file_id = 0;

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get form data
    $file_number = $_POST['file_number'];
    $new_all_in = $_POST['new_all_in'];
    
    // Validate input
    if (empty($file_number)) {
        $_SESSION['error_message'] = "File number is required.";
        header('Location: update_all_in.php');
        exit();
    }
    
    if (!is_numeric($new_all_in) || $new_all_in <= 0) {
        $_SESSION['error_message'] = "All-In amount must be a positive number.";
        header('Location: update_all_in.php?file_number=' . urlencode($file_number));
        exit();
    }
    
    try {
        // Begin transaction
        $conn->begin_transaction();
        
        // Get the total of non-operational costs
        $sql_non_op = "SELECT SUM(charge) as total FROM pre_proforma 
                      WHERE file_number = ? AND category = 'Non-Operational'";
        $stmt_non_op = $conn->prepare($sql_non_op);
        $stmt_non_op->bind_param('s', $file_number);
        $stmt_non_op->execute();
        $result_non_op = $stmt_non_op->get_result();
        $non_op_costs = $result_non_op->fetch_assoc()['total'] ?: 0;
        
        // Calculate new agency fee (All-In - Non-Operational costs)
        $new_agency_fee = $new_all_in - $non_op_costs;
        
        // Check if Pre-Proforma exists for this file
        $check_sql = "SELECT COUNT(*) as count FROM pre_proforma WHERE file_number = ?";
        $check_stmt = $conn->prepare($check_sql);
        $check_stmt->bind_param('s', $file_number);
        $check_stmt->execute();
        $has_pre_proforma = $check_stmt->get_result()->fetch_assoc()['count'] > 0;
        
        if ($has_pre_proforma) {
            // Update the all-in amount in the all_in record if it exists
            $update_all_in_sql = "UPDATE pre_proforma SET all_in_amount = ? 
                                 WHERE file_number = ? AND all_in = 'Yes'";
            $update_all_in_stmt = $conn->prepare($update_all_in_sql);
            $update_all_in_stmt->bind_param('ds', $new_all_in, $file_number);
            $update_all_in_stmt->execute();
            
            // Check if we updated any records - if not, we need to create the all-in record
            if ($update_all_in_stmt->affected_rows == 0) {
                // Get the currency from the file
                $currency_sql = "SELECT currency FROM files WHERE file_number = ?";
                $currency_stmt = $conn->prepare($currency_sql);
                $currency_stmt->bind_param('s', $file_number);
                $currency_stmt->execute();
                $currency = $currency_stmt->get_result()->fetch_assoc()['currency'] ?: '$';
                
                // Insert the all-in record
                $insert_all_in_sql = "INSERT INTO pre_proforma (
                                        file_number, all_in, all_in_amount, category, cost_code, 
                                        cost_description, charge, vatable, vat_amount, currency
                                    ) VALUES (
                                        ?, 'Yes', ?, 'Operational', '1-101', 
                                        'Agency Fee', 0, 'No', 0, ?
                                    )";
                $insert_all_in_stmt = $conn->prepare($insert_all_in_sql);
                $insert_all_in_stmt->bind_param('sds', $file_number, $new_all_in, $currency);
                $insert_all_in_stmt->execute();
            }
            
            // Update the agency fee
            $update_agency_sql = "UPDATE pre_proforma SET charge = ? 
                                 WHERE file_number = ? AND category = 'Operational' 
                                 AND cost_code = '1-101'";
            $update_agency_stmt = $conn->prepare($update_agency_sql);
            $update_agency_stmt->bind_param('ds', $new_agency_fee, $file_number);
            $update_agency_stmt->execute();
            
            // Check if we updated any agency fee records - if not, we need to create it
            if ($update_agency_stmt->affected_rows == 0) {
                // Insert the agency fee record
                $insert_agency_sql = "INSERT INTO pre_proforma (
                                        file_number, category, cost_code, cost_description, 
                                        charge, vatable, vat_amount, currency
                                    ) VALUES (
                                        ?, 'Operational', '1-101', 'Agency Fee', 
                                        ?, 'No', 0, ?
                                    )";
                $insert_agency_stmt = $conn->prepare($insert_agency_sql);
                $insert_agency_stmt->bind_param('sds', $file_number, $new_agency_fee, $currency);
                $insert_agency_stmt->execute();
            }
        } else {
            // No pre-proforma exists yet - we need to create both the all-in and agency fee records
            // Get currency from the file
            $currency_sql = "SELECT currency FROM files WHERE file_number = ?";
            $currency_stmt = $conn->prepare($currency_sql);
            $currency_stmt->bind_param('s', $file_number);
            $currency_stmt->execute();
            $currency = $currency_stmt->get_result()->fetch_assoc()['currency'] ?: '$';
            
            // Insert the all-in record
            $insert_all_in_sql = "INSERT INTO pre_proforma (
                                    file_number, all_in, all_in_amount, category, cost_code, 
                                    cost_description, charge, vatable, vat_amount, currency
                                ) VALUES (
                                    ?, 'Yes', ?, 'Operational', '1-101', 
                                    'Agency Fee', ?, 'No', 0, ?
                                )";
            $insert_all_in_stmt = $conn->prepare($insert_all_in_sql);
            $insert_all_in_stmt->bind_param('sdds', $file_number, $new_all_in, $new_agency_fee, $currency);
            $insert_all_in_stmt->execute();
        }
        
        // Add to pending approvals table
        $check_pending_sql = "SELECT id FROM pending_approvals WHERE file_number = ?";
        $check_pending_stmt = $conn->prepare($check_pending_sql);
        $check_pending_stmt->bind_param('s', $file_number);
        $check_pending_stmt->execute();
        $has_pending = $check_pending_stmt->get_result()->num_rows > 0;
        
        if ($has_pending) {
            // Update existing pending approval
            $update_pending_sql = "UPDATE pending_approvals SET 
                                   new_all_in_amount = ?, 
                                   proposed_by = ?, 
                                   proposed_date = NOW(), 
                                   approved = 'Pending', 
                                   approver = NULL, 
                                   approved_date = NULL 
                                   WHERE file_number = ?";
            $update_pending_stmt = $conn->prepare($update_pending_sql);
            $update_pending_stmt->bind_param('dss', $new_all_in, $full_name, $file_number);
            $update_pending_stmt->execute();
        } else {
            // Create new pending approval
            $insert_pending_sql = "INSERT INTO pending_approvals (
                                    file_number, new_all_in_amount, proposed_by, proposed_date
                                  ) VALUES (?, ?, ?, NOW())";
            $insert_pending_stmt = $conn->prepare($insert_pending_sql);
            $insert_pending_stmt->bind_param('sds', $file_number, $new_all_in, $full_name);
            $insert_pending_stmt->execute();
        }
        
        // Commit the transaction
        $conn->commit();
        
        $_SESSION['success_message'] = "All-In amount updated successfully. Pending approval.";
        header('Location: all_in_list.php');
        exit();
        
    } catch (Exception $e) {
        // Rollback transaction on error
        $conn->rollback();
        
        $_SESSION['error_message'] = "Error updating All-In amount: " . $e->getMessage();
        header('Location: update_all_in.php?file_number=' . urlencode($file_number));
        exit();
    }
} else {
    // Check if file number is provided
    if (isset($_GET['file_number']) && !empty($_GET['file_number'])) {
        $file_number = $_GET['file_number'];
        
        // Get file details
        $sql_file = "SELECT f.*, c.first_name as customer_name 
                    FROM files f 
                    LEFT JOIN customers c ON f.customer = c.customer_id 
                    WHERE f.file_number = ?";
        $stmt_file = $conn->prepare($sql_file);
        $stmt_file->bind_param('s', $file_number);
        $stmt_file->execute();
        $result_file = $stmt_file->get_result();
        
        if ($result_file->num_rows > 0) {
            $file_details = $result_file->fetch_assoc();
            $file_id = $file_details['id'];
            $currency = $file_details['currency'];
            
            // Get existing all-in amount
            $sql_all_in = "SELECT all_in_amount FROM pre_proforma 
                          WHERE file_number = ? AND all_in = 'Yes' 
                          LIMIT 1";
            $stmt_all_in = $conn->prepare($sql_all_in);
            $stmt_all_in->bind_param('s', $file_number);
            $stmt_all_in->execute();
            $result_all_in = $stmt_all_in->get_result();
            
            if ($result_all_in->num_rows > 0) {
                $existing_all_in = $result_all_in->fetch_assoc()['all_in_amount'];
            }
            
            // Get non-operational costs
            $sql_non_op = "SELECT cost_code, cost_description, charge 
                          FROM pre_proforma 
                          WHERE file_number = ? AND category = 'Non-Operational'
                          ORDER BY id";
            $stmt_non_op = $conn->prepare($sql_non_op);
            $stmt_non_op->bind_param('s', $file_number);
            $stmt_non_op->execute();
            $result_non_op = $stmt_non_op->get_result();
            
            while ($row = $result_non_op->fetch_assoc()) {
                $all_non_op_costs[] = $row;
                $non_op_costs += $row['charge'];
            }
        } else {
            $_SESSION['error_message'] = "File not found.";
            header('Location: all_in_list.php');
            exit();
        }
    }
}

// Get system name from settings
$sql_settings = "SELECT setting_value FROM settings WHERE setting_key = 'system_name'";
$result_settings = $conn->query($sql_settings);
if ($result_settings && $result_settings->num_rows > 0) {
    $row_settings = $result_settings->fetch_assoc();
    $system_name = $row_settings['setting_value'];
} else {
    $system_name = "Royal Freight Management System";
}
?>

<?php include 'layouts/main.php'; ?>
<head>
    <?php includeFileWithVariables('layouts/title-meta.php', array('title' => 'Update All-In Amount')); ?>
    <?php include 'layouts/head-css.php'; ?>
    
    <!-- Select2 CSS -->
    <link href="assets/libs/select2/select2.min.css" rel="stylesheet" type="text/css" />
    
    <!-- Sweet Alert css -->
    <link href="assets/libs/sweetalert2/sweetalert2.min.css" rel="stylesheet" type="text/css" />
    
    <style>
        .file-details {
            background-color: #f8f9fa;
            border-radius: 4px;
            padding: 15px;
            margin-bottom: 20px;
        }
        .file-details .label {
            font-weight: 600;
            margin-bottom: 5px;
            color: #495057;
        }
        .form-label {
            font-weight: 500;
        }
        .costs-table {
            margin-top: 15px;
        }
        .total-row {
            font-weight: bold;
            background-color: #f8f9fa;
        }
        .select2-container--default .select2-selection--single {
            height: 38px;
            border-color: #ced4da;
        }
        .select2-container--default .select2-selection--single .select2-selection__rendered {
            line-height: 36px;
        }
        .select2-container--default .select2-selection--single .select2-selection__arrow {
            height: 36px;
        }
        .calculation-box {
            background-color: #f8f9fc;
            border: 1px solid #e9ecef;
            border-radius: 4px;
            padding: 15px;
            margin-top: 20px;
        }
        .calculation-box .row {
            margin-bottom: 10px;
        }
        .calculation-result {
            font-weight: bold;
            font-size: 18px;
            color: #0a58ca;
        }
    </style>
</head>

<body>
    <!-- Begin page -->
    <div id="layout-wrapper">
        <?php include 'layouts/menu.php'; ?>

        <!-- ============================================================== -->
        <!-- Start right Content here -->
        <!-- ============================================================== -->
        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <?php includeFileWithVariables('layouts/page-title.php', array('pagetitle' => 'Shipment Management', 'title' => 'Update All-In Amount')); ?>

                    <?php
                    // Show success message if set
                    if (isset($_SESSION['success_message'])) {
                        echo '<div class="alert alert-success alert-dismissible fade show" role="alert">';
                        echo $_SESSION['success_message'];
                        echo '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
                        echo '</div>';
                        unset($_SESSION['success_message']);
                    }
                    
                    // Show error message if set
                    if (isset($_SESSION['error_message'])) {
                        echo '<div class="alert alert-danger alert-dismissible fade show" role="alert">';
                        echo $_SESSION['error_message'];
                        echo '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
                        echo '</div>';
                        unset($_SESSION['error_message']);
                    }
                    ?>

                    <div class="row">
                        <div class="col-lg-12">
                            <div class="card">
                                <div class="card-header">
                                    <h4 class="card-title mb-0">
                                        <?php echo empty($file_number) ? 'Update All-In Amount' : 'Update All-In Amount for File: ' . $file_number; ?>
                                    </h4>
                                </div>
                                <div class="card-body">
                                    <?php if (empty($file_number)): ?>
                                    <!-- File selection form -->
                                    <form method="GET" action="">
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <label for="file-select" class="form-label">Select File</label>
                                                <select class="form-control select2" id="file-select" name="file_number" required>
                                                    <option value="">Select a file...</option>
                                                    <?php
                                                    // Get files that have pre-proforma entries
                                                    $sql_files = "SELECT DISTINCT f.file_number, f.consignee_name, f.destination 
                                                                 FROM files f 
                                                                 LEFT JOIN pending_approvals pa ON f.file_number = pa.file_number
                                                                 WHERE 
                                                                    (pa.id IS NULL OR pa.approved != 'Approved') AND
                                                                    NOT EXISTS (
                                                                        SELECT 1 FROM proforma pr WHERE pr.file_number = f.file_number
                                                                    )
                                                                 ORDER BY f.created_date DESC";
                                                    $result_files = $conn->query($sql_files);
                                                    
                                                    if ($result_files->num_rows > 0) {
                                                        while ($row = $result_files->fetch_assoc()) {
                                                            echo "<option value='" . $row['file_number'] . "'>" . 
                                                                $row['file_number'] . " - " . $row['consignee_name'] . 
                                                                " (" . $row['destination'] . ")</option>";
                                                        }
                                                    }
                                                    ?>
                                                </select>
                                            </div>
                                            <div class="col-md-6 d-flex align-items-end">
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="ri-search-line align-bottom me-1"></i> Select File
                                                </button>
                                            </div>
                                        </div>
                                    </form>
                                    <?php elseif ($file_details): ?>
                                    <!-- File details and all-in update form -->
                                    <div class="file-details">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <div class="label">File Number:</div>
                                                    <div><?php echo $file_details['file_number']; ?></div>
                                                </div>
                                                <div class="mb-3">
                                                    <div class="label">Customer:</div>
                                                    <div><?php echo $file_details['customer_name'] ?: 'N/A'; ?></div>
                                                </div>
                                                <div class="mb-0">
                                                    <div class="label">Destination:</div>
                                                    <div><?php echo $file_details['destination']; ?></div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <div class="label">Consignee:</div>
                                                    <div><?php echo $file_details['consignee_name']; ?></div>
                                                </div>
                                                <div class="mb-3">
                                                    <div class="label">Created Date:</div>
                                                    <div><?php echo formatDate($file_details['created_date']); ?></div>
                                                </div>
                                                <div class="mb-0">
                                                    <div class="label">Currency:</div>
                                                    <div><?php echo $file_details['currency']; ?></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <h5 class="mt-4 mb-3">Non-Operational Costs</h5>
                                    <div class="table-responsive costs-table">
                                        <table class="table table-bordered">
                                            <thead class="table-light">
                                                <tr>
                                                    <th>Cost Code</th>
                                                    <th>Description</th>
                                                    <th class="text-end">Amount (<?php echo $currency; ?>)</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php if (!empty($all_non_op_costs)): ?>
                                                    <?php foreach ($all_non_op_costs as $cost): ?>
                                                    <tr>
                                                        <td><?php echo $cost['cost_code']; ?></td>
                                                        <td><?php echo $cost['cost_description']; ?></td>
                                                        <td class="text-end"><?php echo number_format($cost['charge'], 2); ?></td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                <?php else: ?>
                                                    <tr>
                                                        <td colspan="3" class="text-center">No non-operational costs found</td>
                                                    </tr>
                                                <?php endif; ?>
                                                <tr class="total-row">
                                                    <td colspan="2" class="text-end">Total Non-Operational Costs:</td>
                                                    <td class="text-end"><?php echo number_format($non_op_costs, 2); ?></td>
                                                </tr>
                                            </tbody>
                                        </table>
                                    </div>
                                    
                                    <form method="POST" action="" id="update-form">
                                        <input type="hidden" name="file_number" value="<?php echo $file_number; ?>">
                                        
                                        <div class="row mt-4">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="existing-all-in" class="form-label">Existing All-In Amount (<?php echo $currency; ?>):</label>
                                                    <input type="text" class="form-control" id="existing-all-in" value="<?php echo number_format($existing_all_in, 2); ?>" readonly>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label for="new-all-in" class="form-label">New All-In Amount (<?php echo $currency; ?>):</label>
                                                    <input type="number" class="form-control" id="new-all-in" name="new_all_in" step="0.01" min="0" required 
                                                           value="<?php echo $existing_all_in; ?>"
                                                           onchange="calculateAgencyFee()">
                                                    <div class="form-text">Enter the new All-In amount. This will update the Agency Fee automatically.</div>
                                                </div>
                                            </div>
                                            
                                            <div class="col-md-6">
                                                <div class="calculation-box">
                                                    <h6 class="mb-3">Agency Fee Calculation</h6>
                                                    
                                                    <div class="row">
                                                        <div class="col-6">New All-In Amount:</div>
                                                        <div class="col-6 text-end" id="calc-all-in"><?php echo $currency . ' ' . number_format($existing_all_in, 2); ?></div>
                                                    </div>
                                                    
                                                    <div class="row">
                                                        <div class="col-6">Non-Operational Costs:</div>
                                                        <div class="col-6 text-end" id="calc-non-op"><?php echo $currency . ' ' . number_format($non_op_costs, 2); ?></div>
                                                    </div>
                                                    
                                                    <hr>
                                                    
                                                    <div class="row">
                                                        <div class="col-6">New Agency Fee:</div>
                                                        <div class="col-6 text-end calculation-result" id="calc-agency-fee">
                                                            <?php echo $currency . ' ' . number_format($existing_all_in - $non_op_costs, 2); ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="row mt-4">
                                            <div class="col-12 text-end">
                                                <a href="all_in_list.php" class="btn btn-secondary me-2">Cancel</a>
                                                <button type="submit" class="btn btn-primary" id="submit-btn">Update All-In Amount</button>
                                            </div>
                                        </div>
                                    </form>
                                    <?php else: ?>
                                    <div class="alert alert-warning">
                                        <i class="ri-alert-line me-2"></i>
                                        File not found or no file selected. Please select a valid file.
                                    </div>
                                    <div class="text-center mt-3">
                                        <a href="all_in_list.php" class="btn btn-primary">Back to All-In List</a>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <!-- container-fluid -->
            </div>
            <!-- End Page-content -->

            <?php include 'layouts/footer.php'; ?>
        </div>
        <!-- end main content-->
    </div>
    <!-- END layout-wrapper -->

    <?php include 'layouts/vendor-scripts.php'; ?>
    
    <!-- Select2 js -->
    <script src="assets/libs/select2/select2.min.js"></script>
    
    <!-- Sweet Alerts js -->
    <script src="assets/libs/sweetalert2/sweetalert2.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Initialize Select2
            $('.select2').select2({
                width: '100%'
            });
            
            // Form submission confirmation
            $('#update-form').on('submit', function(e) {
                e.preventDefault();
                
                Swal.fire({
                    title: 'Confirm Update',
                    text: "Are you sure you want to update the All-In amount? This will require approval before taking effect.",
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonColor: '#3085d6',
                    cancelButtonColor: '#d33',
                    confirmButtonText: 'Yes, update it!'
                }).then((result) => {
                    if (result.isConfirmed) {
                        this.submit();
                    }
                });
            });
            
            // Show success message with SweetAlert if present
            <?php if (isset($_SESSION['swal_success'])): ?>
            Swal.fire({
                icon: 'success',
                title: 'Success!',
                text: '<?php echo $_SESSION['swal_success']; ?>',
                confirmButtonColor: '#3085d6'
            });
            <?php unset($_SESSION['swal_success']); endif; ?>
            
            // Show error message with SweetAlert if present
            <?php if (isset($_SESSION['swal_error'])): ?>
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: '<?php echo $_SESSION['swal_error']; ?>',
                confirmButtonColor: '#3085d6'
            });
            <?php unset($_SESSION['swal_error']); endif; ?>
        });
        
        // Calculate agency fee based on all-in amount
        function calculateAgencyFee() {
            const newAllIn = parseFloat(document.getElementById('new-all-in').value) || 0;
            const nonOpCosts = <?php echo $non_op_costs; ?>;
            const currency = '<?php echo $currency; ?>';
            
            // Calculate agency fee
            const agencyFee = newAllIn - nonOpCosts;
            
            // Update the calculation box
            document.getElementById('calc-all-in').textContent = currency + ' ' + newAllIn.toFixed(2);
            document.getElementById('calc-non-op').textContent = currency + ' ' + nonOpCosts.toFixed(2);
            document.getElementById('calc-agency-fee').textContent = currency + ' ' + agencyFee.toFixed(2);
            
            // Disable submit button if agency fee is negative
            const submitBtn = document.getElementById('submit-btn');
            if (agencyFee < 0) {
                submitBtn.disabled = true;
                submitBtn.title = 'Agency fee cannot be negative';
                Swal.fire({
                    icon: 'warning',
                    title: 'Invalid Amount',
                    text: 'The All-In amount must be greater than the total non-operational costs.',
                    confirmButtonColor: '#3085d6'
                });
            } else {
                submitBtn.disabled = false;
                submitBtn.title = '';
            }
        }
    </script>

    <!-- App js -->
    <script src="assets/js/app.js"></script>
</body>
</html>